provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-dev"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-dev/"
}

variable "service_name" { default = "tourney-passport" }
variable "service_env" { default = "staging" }

data "terraform_remote_state" "dev" {
  backend = "s3"
  config {
    bucket = "twitch-esports-dev"
    key = "tfstate/esports/terraform/dev/raw"
    profile = "twitch-esports-dev"
    region = "us-east-1"
  }
}

resource "aws_db_subnet_group" "default" {
  name = "${var.service_name}"
  subnet_ids = ["${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "subnet_ids"))}"]
}

resource "aws_security_group" "db" {
  name = "${var.service_name}-db"
  description = "RDS for ${var.service_name}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "vpc_id")}"
  ingress {
    from_port = "5432"
    to_port = "5432"
    protocol = "tcp"
    cidr_blocks = ["10.0.0.0/8"]
  }
  egress {
    from_port = 0
    to_port = 0
    protocol = "-1"
    cidr_blocks = ["10.0.0.0/8"]
  }
}

resource "aws_db_instance" "default" {
  allocated_storage = 15
  engine = "postgres"
  engine_version = "9.5.4"
  instance_class = "db.t2.micro"
  name = "passport"
  username = "tourney"
  password = "tourn3yLUL"
  db_subnet_group_name = "${aws_db_subnet_group.default.name}"
  vpc_security_group_ids = ["${aws_security_group.db.id}"]
}

resource "aws_ecr_repository" "tourney-passport" {
  name = "tourney-passport"
}

resource "template_file" "passport_task_definition" {
  template = "${file("task-definitions/tourney-passport.json")}"
  vars {
    name = "tourney-passport"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.tourney-passport.repository_url, "https://", "")}:latest"
    db_url = "postgres://${aws_db_instance.default.username}:${aws_db_instance.default.password}@${aws_db_instance.default.endpoint}/${aws_db_instance.default.name}"
    container_port = 3000
    cpu = 256
    mem = 256
  }
}

resource "aws_iam_role" "task" {
  name = "ecs_${var.service_name}_task_role"
  assume_role_policy = <<EOF
{
  "Version": "2008-10-17",
  "Statement": [{
    "Action": "sts:AssumeRole",
    "Principal": {
      "Service": [
        "ecs-tasks.amazonaws.com"
      ]
    },
    "Effect": "Allow"
  }]
}
EOF
}

resource "aws_iam_role_policy" "task" {
  name = "ecs_${var.service_name}_task_role"
  role = "${aws_iam_role.task.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": ["kinesis:*"],
      "Resource": ["*"]
    }
  ]
}
EOF
}

resource "aws_ecs_task_definition" "passport_task" {
  family = "tourney-passport"
  container_definitions = "${template_file.passport_task_definition.rendered}"
  task_role_arn = "${aws_iam_role.task.arn}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-05-18"
  name = "tourney-passport"
  prefix = "tourney_passport"
  container_port = "3000"
  cluster = "tourney"
  env = "staging"
  alb = true
  rails_json = true
  desired_count = 3
  certificate_arn = "${module.common.service_certificate_external}"
  task_definition = "${aws_ecs_task_definition.passport_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.dev.alarm_sns_topic_arn}"
}

output "ecr_url" {
  value = "${aws_ecr_repository.tourney-passport.repository_url}"
}
