import * as stringify from 'json-stringify-safe';
import { API_FETCH } from '../actions';
import { APIAction } from './action';

export interface APIParams {
  [key: string]: any;
}

export interface APIRequestOptions {
  query?: Object;
  method?: string;
  body?: string | FormData;
  headers?: Object;
}

export interface APIRequest {
  url: string;
  options?: APIRequestOptions;
  params?: any; // How would I allow interfaces that inherit APIParams here?
  object?: any;
}

export interface APIActionSet {
  pre?: Function;
  success?: Function;
  redirected?: Function;
  error?: Function;
}

export class APIHandler {
  public dispatch: Function;

  public constructor(dispatch: any) {
    this.dispatch = dispatch;
  }

  public call(request: APIRequest, actionSet: APIActionSet): APIAction {
    request.options = Object.assign({
      headers: {
        'Content-Type': 'application/json',
        Accept: 'application/js'
      }
    }, request.options);

    if (request.params) {
      const params = decycle(request.params);
      let query = Object.keys(params)
        .map( (k) => {
          if (params[k] !== '') {
            return [encodeURIComponent(k), encodeURIComponent(params[k])].join('=');
          }
        })
        .filter( (q) => q !== undefined )
        .join('&');

      if (query.length > 0) {
        request.url += '?' + query;
      }
    }

    if (request.object) {
      request.options.body = JSON.stringify(decycle(request.object));
    }

    if (typeof actionSet.pre === 'function') {
      this.dispatch(actionSet.pre());
    }

    return new APIAction({
      handler: this,
      request
    });
  }
}

export function decycle(object: any) {
  return JSON.parse(stringify(object, undefined, 2, () => ''));
}

export function createFormData(object: any, form?: FormData, namespace?: string): FormData {
  const formData = form || new FormData();
  for (let property in object) {
    if (!object.hasOwnProperty(property) || !object[property]) {
      continue;
    }

    const formKey = namespace ? `${namespace}[${property}]` : property;
    if (object[property] instanceof Date) {
      formData.append(formKey, object[property].toISOString());
    } else if (typeof object[property] === 'object' && Array.isArray(object[property])) {
      // send empty array if necessary
      if (object[property].length === 0) {
        formData.append(property, '');
      }
      object[property].forEach((obj: any) => {
        formData.append(`${property}[]`, obj);
      });
    } else if (typeof object[property] === 'object' && !(object[property] instanceof File)) {
      createFormData(decycle(object[property]), formData, formKey);
    } else {
      formData.append(formKey, object[property]);
    }
  }
  return formData;
}

export * from './action';
export * from './user';
export * from './team';
export * from './teamInvitation';
export * from './tournament';
export * from './tournamentEntry';
export * from './game';
export * from './match';
export * from './series';
export * from './article';
export * from './images';
export * from './season';
export * from './event';
export * from './schedule';
