import { normalize } from 'normalizr';

import * as UserActions from '../actions/user';
import {
  createFormData,
  APIHandler,
  APIRequest,
  APIActionSet,
  APIParams,
  APIAction
} from '../api';
import { config } from '../config';
import { UserModel } from '../models/user';

interface QueryParams {
  [key: string]: any;
  token: string;
}

export function getUsers() {
  let request: APIRequest = {
    url: config.apiUrl + '/v1/users'
  };
  let actions: APIActionSet = {};

  return (dispatch: any): APIAction => {
    const api = new APIHandler(dispatch);
    const action = api.call(request, actions);
    action.promise.then((json) => {
      dispatch(UserActions.createUpdateUsersAction(json.data));
    });
    return action;
  };
}

export function getUser(id: string) {
  let request: APIRequest = {
    url: `${config.apiUrl}/v1/users/${id}`
  };
  let actions: APIActionSet = {};

  return (dispatch: any) => {
    const api = new APIHandler(dispatch);
    const action = api.call(request, actions);
    action.promise.then((json) => {
      dispatch(UserActions.createUpdateUsersAction([json.data]));
    });
    return action;
  };
}

interface AuthUserCallbackParams extends APIParams {
  token: string;
  code: string;
}

// Handles all user authentication and should probably be broken up
//
// When given a token and a code, it will generate an async action that simply
// passes on an OAuth2 callback.
//
// When not given any arguments, it will generate an async action that
// initiates an OAuth2 request and redirects to the provider.
export function authUser(token: string = '', code: string = '', redirect: boolean = true) {
  let request: APIRequest;
  let actions: APIActionSet = {};

  if (token === '' || code === '') {
    request = {
      url: config.apiUrl + '/v1/users/auth'
    };
  } else {
    request = {
      url: config.apiUrl + '/v1/users/auth/callback/twitch'
    };
    request.params = {
      token,
      code
    } as AuthUserCallbackParams;
  }

  return (dispatch: any) => {
    const api = new APIHandler(dispatch);
    const action = api.call(request, actions);
    action.promise.then((json) => {
      if (request.params && request.params.token) {
        localStorage.setItem('token', request.params.token);
      }

      if (json.status === 302 && redirect) {
        window.location.href = json.redirect;
        return json;
      } else if (json.data) {
        dispatch(UserActions.createAuthUserAction(json.data));
      }
    });
    return action;
  };
}

export function updateUser(user: UserModel) {
  const formData = createFormData(user);
  let request: APIRequest = {
    url: config.apiUrl + '/v1/users/' + user.id,
    options: {
      headers: {},
      method: 'PATCH',
      body: formData
    }
  };

  let actions: APIActionSet = {};

  return (dispatch: any) => {
    const api = new APIHandler(dispatch);
    const action = api.call(request, actions);
    action.promise.then((json) => {
      dispatch(UserActions.createUpdateUsersAction([json.data]));
    });
    return action;
  };
}

export function verifyEmailToken(token: string) {
  const request: APIRequest = {
    url: config.apiUrl + '/v1/users/verify/' + token
  };

  const actions: APIActionSet = {};

  return (dispatch: any) => {
    const api = new APIHandler(dispatch);
    const action = api.call(request, actions);
    action.promise.then((json) => {
      dispatch(UserActions.createUpdateUsersAction([json.data]));
    });
    return action;
  };
}

export function logoutUser() {
  return (dispatch: any) => {
    localStorage.removeItem('token');
    dispatch(UserActions.createLogoutAction());
    return Promise.resolve();
  };
}
