import * as _ from 'lodash';
import { normalize } from 'normalizr';

import * as actions from './actions';
import * as models from './models';
import * as schemas from './schemas';

export type UserStore = {
  [id: string]: models.UserModel | string;
  current?: string;
};

export type TeamStore = {
  [id: string]: models.TeamModel
};

export type TeamInvitationStore = {
  [id: string]: models.TeamInvitationModel
};

export type TournamentStore = {
  [id: string]: models.TournamentModel
};

export type TournamentEntryStore = {
  [id: string]: models.TournamentEntryModel
};

export type TournamentStageStore = {
  [id: string]: models.TournamentStageModel
};

export type GameStore = {
  [id: string]: models.GameModel | string;
};

export type MatchStore = {
  [id: string]: models.MatchModel | string;
};

export type SeriesStore = {
  [id: string]: models.SeriesModel | string;
};

export type OpponentStore = {
  [id: string]: models.OpponentModel | string;
};

export type ArticleStore = {
  [id: string]: models.ArticleModel
};

export type SeasonStore = {
  [id: string]: models.SeasonModel
};

export type EventStore = {
  [id: string]: models.EventModel
};

export type TourneyState = {
  users: UserStore,
  teams: TeamStore,
  team_invitations: TeamInvitationStore,
  tournaments: TournamentStore,
  tournament_entries: TournamentEntryStore,
  tournament_stages: TournamentStageStore,
  games: GameStore,
  matches: MatchStore,
  series: SeriesStore,
  opponents: OpponentStore,
  articles: ArticleStore,
  seasons: SeasonStore,
  events: EventStore,
  requests: any,
  queries: any
};

export const DefaultState: TourneyState = {
  users: {},
  teams: {},
  team_invitations: {},
  tournaments: {},
  tournament_entries: {},
  tournament_stages: {},
  games: {},
  matches: {},
  series: {},
  opponents: {},
  articles: {},
  seasons: {},
  events: {},
  requests: {},
  queries: {}
};

function mergeState(state: TourneyState, newState: TourneyState) {
  return Object.assign({}, _.mergeWith(state, newState, (objValue, srcValue) => {
    if (_.isArray(objValue)) {
      return srcValue;
    }
  }));
}

export function reducer(state: TourneyState = DefaultState, action: any): Object {
  let newState: any;

  if (action.meta) {
    if (action.meta.fetch && action.meta.fetch.url) {
      state = { ...state, requests: { ...state.requests,
        [action.meta.fetch.url]: state.requests[action.meta.fetch.url] + 1 || 1 } };
    }

    if (action.meta.fetch.done) {
      state = { ...state };
      const url = new URL(action.meta.fetch.url);
      // searchParams will currently throw an error, fixed here:
      // https://github.com/Microsoft/TSJS-lib-generator/pull/198
      if ('searchParams' in HTMLAnchorElement.prototype) {
      const cacheKey = url.searchParams.get('cache');
        if (cacheKey && cacheKey !== '') {
          state.queries[cacheKey] = action.payload;
          if (action.payload.data instanceof Array) {
            state.queries[cacheKey].idMap = action.payload.data.map((o: any) => o.id);
          }
        }
      }
      delete state.requests[action.meta.fetch.url];
    }
  }

  switch (action.type) {
    case actions.AUTH_USER:
      const normalized = normalize(action.payload, schemas.UserSchema).entities;
      normalized.users.current = action.payload.id;
      return mergeState(state, normalized);

    case actions.UPDATE_USERS:
      return mergeState(state, normalize(action.payload,
                                         [schemas.UserSchema]).entities);
    case actions.LOGOUT_USER:
      newState = { ...state };
      delete newState.users.current;
      return newState;

    case actions.UPDATE_TEAMS:
      return mergeState(state, normalize(action.payload,
                                         [schemas.TeamSchema]).entities);

    case actions.UPDATE_TOURNAMENTS:
      return mergeState(state, normalize(action.payload,
                                         [schemas.TournamentSchema]).entities);

    case actions.UPDATE_TOURNAMENT_ENTRIES:
      return mergeState(state, normalize(action.payload,
                                         [schemas.TournamentEntrySchema]).entities);

    case actions.UPDATE_TEAM_INVITATIONS:
      return mergeState(state, normalize(action.payload,
                                         [schemas.TeamInvitationSchema]).entities);

    case actions.UPDATE_GAMES:
      return mergeState(state, normalize(action.payload,
                                         [schemas.GameSchema]).entities);

    case actions.UPDATE_MATCHES:
      return mergeState(state, normalize(action.payload,
                                         [schemas.MatchSchema]).entities);

    case actions.UPDATE_EVENTS:
      return mergeState(state, normalize(action.payload,
                                         [schemas.EventSchema]).entities);

    case actions.UPDATE_SCHEDULE:
      return mergeState(state, normalize(action.payload,
                                         [schemas.ScheduledEvent]).entities);

    case actions.DELETE_EVENT:
      newState = mergeState(state, normalize(action.payload,
                                        [schemas.EventSchema]).entities);
      delete newState.events[action.payload.id];
      return newState;

    case actions.UPDATE_SERIES:
      return mergeState(state, normalize(action.payload,
                                         [schemas.SeriesSchema]).entities);

    case actions.UPDATE_ARTICLES:
      return mergeState(state, normalize(action.payload,
                                        [schemas.ArticleSchema]).entities);

    case actions.DELETE_ARTICLE:
      newState = mergeState(state, normalize(action.payload,
                                        [schemas.ArticleSchema]).entities);
      delete newState.articles[action.payload.id];
      return newState;

    case actions.UPDATE_SEASONS:
      return mergeState(state, normalize(action.payload,
                                         [schemas.SeasonSchema]).entities);

    // Why merge the response?
    // we expect the deleted invitation in the response, which includes updated
    // associations for team_id and team that we want to merge into the state.
    case actions.DELETE_TEAM_INVITATIONS:
      newState = mergeState(state, normalize(action.payload,
                                                 [schemas.TeamInvitationSchema]).entities);
      action.payload.map((invitation: models.TeamInvitationModel) =>
                         delete newState.team_invitations[invitation.id]);
      return newState;

    case actions.DELETE_TOURNAMENT_ENTRIES:
      newState = mergeState(state, normalize(action.payload,
                                                 [schemas.TournamentEntrySchema]).entities);
      action.payload.map((entry: models.TournamentEntryModel) =>
                         delete newState.tournament_entries[entry.id]);
      return newState;

    default:
      return state;
  }
}
