// For more information about `relationshipStrategy`, see RFC #1:
// https://git-aws.internal.justin.tv/esports/tourney-sdk-react/issues/19
import { isArray } from 'lodash';

import { schema, denormalize } from 'normalizr';
import { connect } from 'react-redux';

import {
  getUser,
  getTeam,
  getTournament,
  getTournamentEntries,
  APIAction
} from './api';
import { config } from './config';
import {
  TeamModel,
  UserModel
} from './models';

// If we have an ID map, but no objects, merge the map into the relation key
// and let the relationshipStrategy callback decide what to load.
export function mergeIds(object: any, relationKey: string, idsKey: string): Boolean {
  if (object[relationKey] === undefined && object[idsKey] !== undefined) {
    object[relationKey] = object[idsKey];
    return object[idsKey].length > 0 ? true : false;
  }
  return false;
}

export function relationshipStrategy(mapping: any, loadMissing: Function) {
  return (value: any, parent: any, key: any) => {
    let result = { ...value };
    let missing = [];
    for (let relationKey of Object.keys(mapping)) {
      if (mergeIds(result, relationKey, mapping[relationKey])) {
        missing.push(relationKey);
      }
    }
    loadMissing.call(this, result, missing);
    return result;
  };
}

export function fetchMissing(entitySchema: schema.Entity, object: any, relation: string) {
  const dispatch = config.store.dispatch;

  // TODO We'll need this for now. Don't try to clean up side effects, but
  // focus on implementing an approach that doesn't require them.
  setTimeout(() => {
    switch (entitySchema) {
      case TeamSchema: return dispatch(getTeam(object.id));
      // case UserSchema: return dispatch(getUser(object.id));
      case TournamentSchema: return dispatch(getTournament(object.id));
      case OpponentSchema: return ; // no api point here
      default:
    }

    switch (relation) {
      case 'users':
        for (let userId of object[relation]) {
          dispatch(getUser(userId));
        }
        break;
      case 'tournament_entries':
        dispatch(getTournamentEntries({ ids: object[relation] }));
        break;
      case 'stages':
        dispatch(getTournament(object.id));
        break;
      case 'team_invitations':
        break;
      default:
        return true;
    }
  });
}

async function checkMissing(entitySchema: schema.Entity, object: any, relations: string[]) {
  if (relations.length > 0) {
    const inflated = denormalize(object, entitySchema, config.store.getState().tourney);
    for (let relation of relations) {
      // If any of the objects in the missing relationship have not been loaded
      // reload the object (or dispatch any other action necessary to obtain
      // the data.)
      if (inflated[relation] === undefined ||
          (isArray(inflated[relation]) && inflated[relation].indexOf(undefined) !== -1)) {
        await fetchMissing(entitySchema, object, relation);
      }
    }
  }
}

const TeamSchema: schema.Entity = new schema.Entity('teams', {}, {
  processStrategy: relationshipStrategy({
    tournament_entries: 'tournament_entry_ids',
    team_invitations: 'team_invitation_ids'
    // users: 'user_ids'
  }, (...args: any[]) => {
    return checkMissing.call(undefined, this.TeamSchema, ...args);
  })
});
const TournamentEntrySchema: schema.Entity = new schema.Entity('tournament_entries', {}, {
  processStrategy: relationshipStrategy({
    tournament: 'tournament_id',
    entrant: 'entrant_id'
  }, (...args: any[]) => {
    return checkMissing.call(undefined, this.TournamentEntrySchema, ...args);
  })
});
const UserSchema: schema.Entity = new schema.Entity('users', {}, {
  processStrategy: relationshipStrategy({
    teams: 'team_ids'
  }, (...args: any[]) => {
    return checkMissing.call(undefined, this.UserSchema, ...args);
  })
});
const TournamentSchema: schema.Entity = new schema.Entity('tournaments', {}, {
  processStrategy: relationshipStrategy({
    game: 'game_id',
    tournament_entries: 'tournament_entry_ids',
    stages: 'stage_ids'
  }, (...args: any[]) => {
    return checkMissing.call(undefined, this.TournamentSchema, ...args);
  })
});
const SeasonSchema: schema.Entity = new schema.Entity('seasons', {}, {
  processStrategy: (value, parent, key) => {
    return {
      ...value,
      tournaments: value.tournament_ids
    };
  }
});
const TournamentStageSchema: schema.Entity = new schema.Entity('tournament_stages', {}, {
  processStrategy: (value, parent, key) => {
    return {
      ...value,
      tournament: value.tournament_id
    };
  }
});
const TeamInvitationSchema: schema.Entity = new schema.Entity('team_invitations');
const GameSchema: schema.Entity = new schema.Entity('games');
const EventSchema: schema.Entity = new schema.Entity('events');
const MatchSchema: schema.Entity = new schema.Entity('matches', {});
const SeriesSchema: schema.Entity = new schema.Entity('series', {});
const ArticleSchema: schema.Entity = new schema.Entity('articles', {});
const OpponentSchema: schema.Entity = new schema.Entity('opponents', {}, {
  processStrategy: (value: any, parent: any, key: any) => {
    // TODO: Remove this once we have unions normalizing from api correctly.
    return {
      ...value,
      contender: {
        id: value.contender_id,
        schema: value.contender_type === 'Team' ? 'team' : 'user'
      }
    };
  }
});

const ScheduledEvent = new schema.Union({
  event: EventSchema,
  series: SeriesSchema
}, (value, parent, key) => {
  let result = value.type;
  if (result.indexOf('::') !== -1) {
    result = result.split('::')[0];
  }
  return result.toLowerCase();
});

TeamSchema.define({
  tournament_entries: [TournamentEntrySchema],
  team_invitations: [TeamInvitationSchema],
  users: [UserSchema]
});

TeamInvitationSchema.define({
  team: TeamSchema
});

const UnionSchema = new schema.Union({
  team: TeamSchema,
  user: UserSchema
}, (value, parent, key) => {
  return parent.entrant_type;
});

TournamentEntrySchema.define({
  tournament: TournamentSchema,
  entrant: TeamSchema
});

TournamentSchema.define({
  tournament_entries: [TournamentEntrySchema],
  stages: [TournamentStageSchema],
  game: GameSchema
});

TournamentStageSchema.define({
  tournament: TournamentSchema
});

UserSchema.define({
  teams: [TeamSchema]
});

GameSchema.define({
  tournaments: [TournamentSchema]
});

MatchSchema.define({
  game: GameSchema,
  winner: TeamSchema,
  opponents: [OpponentSchema]
});

SeriesSchema.define({
  matches: [MatchSchema],
  opponents: [OpponentSchema]
});

ArticleSchema.define({
  author: UserSchema
});

SeasonSchema.define({
  tournaments: [TournamentSchema]
});

const ContestSchema = new schema.Union({
  match: MatchSchema,
  series: SeriesSchema
}, (value, parent, key) => parent.contest_type);

// since we are are not normalizing this through SDK,
// unionAttribute key doesn't matter, it doesn't use it on denormalize.
const ContenderSchema = new schema.Union({
  team: TeamSchema,
  user: UserSchema
}, 'unused_variable');

OpponentSchema.define({
  contest: ContestSchema,
  contender: ContenderSchema
});

export {
  UserSchema,
  TeamSchema,
  TeamInvitationSchema,
  TournamentEntrySchema,
  TournamentSchema,
  TournamentStageSchema,
  GameSchema,
  MatchSchema,
  SeriesSchema,
  OpponentSchema,
  ArticleSchema,
  SeasonSchema,
  EventSchema,
  ScheduledEvent
};
