import configureStore from 'redux-mock-store';
import * as nock from 'nock';

import { middleware } from '../../lib/middleware';
const mockStore = configureStore(middleware);

import { config } from '../../lib/config';
config.apiUrl = 'http://example.com';

import * as reducer from '../../lib/reducer';
import * as actions from '../../lib/actions/tournamentEntry';
import * as api from '../../lib/api/tournamentEntry';

describe('api/tournamentEntry', () => {
  describe('updateTournamentEntry', () => {
    it('should accept object w/ circular references', async () => {
      let tournamentEntry = { tournament_id: 0, team_id: 0 };
      tournamentEntry['tournament'] = {
        tournament_entries: [tournamentEntry]
      }
      expect(api.updateTournamentEntry(tournamentEntry)).not.toThrow();
    });
  });

  describe('createOrUpdateTournamentEntry', () => {
    it('should call createTournamentEntry if argument has no id', async () => {
      nock('http://example.com/v1').
        post(`/tournament_entries`).
        reply(200, { status: 200, data: {}});
      const store = mockStore({ tourney: reducer.DefaultState });
      const response = await store.dispatch(api.createOrUpdateTournamentEntry({}));
      // I have given up trying to sanely mock functions in jest+ts, so we'll
      // check the API_FETCH URL instead.
      expect(store.getActions()).toContainEqual({
        type: 'API_FETCH',
        meta: { fetch: { url: 'http://example.com/v1/tournament_entries' } }
      });
    });

    it('should call updateTournamentEntry if argument has id', async () => {
      nock('http://example.com/v1').
        patch(`/tournament_entries/test`).
        reply(200, { status: 200, data: {}});
      const store = mockStore({ tourney: reducer.DefaultState });
      const response = await store.dispatch(api.createOrUpdateTournamentEntry({id: 'test'}));
      expect(store.getActions()).toContainEqual({
        type: 'API_FETCH',
        meta: { fetch: { url: 'http://example.com/v1/tournament_entries/test' } }
      });
    });
  });

  describe('createTournamentEntry', () => {
    it('should create UPDATE_TOURNAMENT_ENTRIES when successful', async () => {
      const tournamentEntry = { tournament_id: 0, team_id: 0 };
      nock('http://example.com/v1').
        post(`/tournament_entries`).
        reply(200, { status: 200, data: tournamentEntry });
      const store = mockStore({ tourney: reducer.DefaultState });
      const response = await store.dispatch(api.createTournamentEntry(tournamentEntry));
      expect(store.getActions()).toContainEqual({
        type: actions.UPDATE_TOURNAMENT_ENTRIES,
        payload: [tournamentEntry]
      });
    });
  });
});
