import configureStore from 'redux-mock-store';
import * as nock from 'nock';

import { middleware } from '../../lib/middleware';
const mockStore = configureStore(middleware);

import { config } from '../../lib/config';
config.apiUrl = 'http://example.com';

import * as reducer from '../../lib/reducer';
import * as actions from '../../lib/actions/user';
import * as api from '../../lib/api/user';

describe('api/user', () => {
  describe('getUsers', () => {
    it('should create UPDATE_USERS when successful', async () => {
      const user = { name: 'Super Mario' };
      nock('http://example.com/v1').
        get('/users').
        reply(200, { status: 200, data: [user] });
      const store = mockStore({ tourney: reducer.DefaultState });
      const response = await store.dispatch(api.getUsers());
      expect(store.getActions()).toContainEqual({
        type: actions.UPDATE_USERS,
        payload: [user]
      });
    });
  });

  describe('authUser', () => {
    beforeEach(() => window.location.href = 'about:blank');

    it('should set window.location for a 302 response', async () => {
      nock('http://example.com/v1').
        get(`/users/auth`).
        reply(200,
              {status: 302, token: 'lel', redirect: 'http://example.com'},
              {'Location': 'http://example.com'});
      const store = mockStore({ tourney: reducer.DefaultState });
      await store.dispatch(api.authUser());
      expect(window.location.href).toEqual('http://example.com');
    });

    it('should not set window.location if redirect is false', async () => {
      // When /callback/:provider fails, it'll come back as if no code/token
      // were supplied, initiating a new oauth cycle.
      nock('http://example.com/v1').
        filteringPath(/\?.*$/g, '').
        get('/users/auth/callback/twitch').
        reply(200,
              {status: 302, token: 'lel', redirect: 'http://example.com'},
              {'Location': 'http://example.com'});
      const store = mockStore({ tourney: reducer.DefaultState });
      await store.dispatch(api.authUser('some-code', 'some-token', false));
      expect(window.location.href).not.toEqual('http://example.com');
    });
  });

  describe('updateUser', () => {
    it('should accept object w/ circular references', async () => {
      let user = { id: 'test-test-test-test', name: 'Super Mario' };
      user['teams'] = [{
        name: 'Circular References Rule',
        users: [user]
      }];
      expect(api.updateUser(user)).not.toThrow();
    });

    it('should create UPDATE_USERS when successful', async () => {
      const user = { id: 'test-test-test-test', name: 'Super Mario' };
      nock('http://example.com/v1').
        patch(`/users/${user.id}`).
        reply(200, { status: 200, data: user });
      const store = mockStore({ tourney: reducer.DefaultState });
      const response = await store.dispatch(api.updateUser(user));
      expect(store.getActions()).toContainEqual({
        type: actions.UPDATE_USERS,
        payload: [user]
      });
    });
  });

  describe('verifyEmailToken', () => {
    it('should create UPDATE_USERS when successful', async () => {
      const user = {
        id: 'test-test-test-test',
        name: 'Super Mario',
        email_token: 'test' };
      nock('http://example.com/v1').
        get(`/users/verify/${user.email_token}`).
        reply(200, { status: 200, data: user });
      const store = mockStore({ tourney: reducer.DefaultState });
      const response = await store.dispatch(api.verifyEmailToken(user.email_token));
      expect(store.getActions()).toContainEqual({
        type: actions.UPDATE_USERS,
        payload: [user]
      });
    });
  });

  describe('logoutUser', () => {
    let store;
    let key = '123';
    beforeEach(() => {
      store = mockStore({ tourney: reducer.DefaultState });
    });

    it('should remove token from localStorage', async () => {
      window.localStorage.setItem('token', key);
      expect(window.localStorage.getItem('token')).toEqual(key);
      await store.dispatch(api.logoutUser());
      expect(window.localStorage.getItem('token')).toBeUndefined();
    });

    it('should create LOGOUT_USER', async () => {
      await store.dispatch(api.logoutUser());
      expect(store.getActions()).toContainEqual({
        type: actions.LOGOUT_USER,
        payload: undefined
      });
    });
  });
});
