import React, { Component } from 'react';
import { Link, NavLink } from 'react-router-dom';
import PropTypes from 'prop-types';
import qs from 'qs';
import { FormattedMessage, injectIntl } from 'react-intl';

import PageHeader from '../components/PageHeader';
import DropdownSelect from '../components/DropdownSelect';
import EventList from './schedule/EventList';
import banner from '../images/mockup-schedule-banner.png'
import icons from '../images/icons.svg';

function search(query) {
  return qs.stringify(query, { encodeValuesOnly: true });
}

function searchToObject(search) {
  var pairs = search.substring(1).split("&"),
    obj = {},
    pair,
    i;

  for ( i in pairs ) {
    if ( pairs[i] === "" ) continue;

    pair = pairs[i].split("=");
    obj[ decodeURIComponent( pair[0] ) ] = decodeURIComponent( pair[1] );
  }

  return obj;
}

function toSnakeCase(str) {
  return str.replace(/[ -]/, '_').toLowerCase();
}

function ToggleLink(props) {
  function isActive(match, location) {
    const values = searchToObject(location.search);

    for (let property in values) {
      if (values.hasOwnProperty(property)
          && values[property] !== undefined
          && toSnakeCase(values[property]) === props.filterValue) {
        return true
      }
    }
    return false
  }

  return (<NavLink {...props} isActive={isActive} activeClassName="is-active" className="o-toggle o-toggle--filter o-h5 c-filter-group__filter" replace />);
}

function toggle(filters, key, value) {
  const values = new Set(filters[key]);

  if (values.has(value)) {
    values.clear();
  } else {
    values.clear();
    values.add(value);
  }

  return { ...filters, [key]: [...values] };
}

function clear(filters, key) {
  const clone = { ...filters };
  delete clone[key];
  return clone;
}

class Schedule extends Component {
  constructor(props) {
    super(props);

    this.state = {
      events: []
    };
  }

  componentDidMount() {
    this.fetchEvents().then((events) => {
      this.setState(() => ({ events }));
    });
  }

  getTypesTitle(query) {
    if (query && query.types && query.types.length === 1) {
      return this.props.intl.formatMessage({id: `schedule.filter.type.${toSnakeCase(query.types[0])}`})
    }
    return this.props.intl.formatMessage({id: 'schedule.filter.type.title'});
  }

  getRegionsTitle(query) {
    if (query && query.regions && query.regions.length === 1) {
      return this.props.intl.formatMessage({id: `schedule.filter.region.${toSnakeCase(query.regions[0])}`})
    }
    return this.props.intl.formatMessage({id: 'schedule.filter.region.title'});
  }

  fetchEvents() {
    // TODO: Replace with actual API call

    return Promise.resolve([
      {
        type: 'EVO',
        title: 'Evo Championship Series',
        region: 'Americas',
        location: 'Las Vegas, NV',
        venue: '',
        address: '',
        startsAt: '2017-07-14',
        endsAt: '2017-07-16',
        flag: 'USA',
        website: 'https://evo.shoryuken.com',
        logo_image: 'evo.svg',
        pot_bonus: '$15,000'
      },
      {
        type: 'Master',
        title: 'CEO',
        region: 'Americas',
        location: 'Orlando, FL',
        venue: '',
        address: '',
        startsAt: '2017-06-16',
        endsAt: '2017-06-18',
        flag: 'USA',
        website: 'https://ceogaming.org/',
        logo_image: 'ceo.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Online',
        title: 'NA East',
        region: 'Americas',
        location: 'Americas',
        venue: '',
        address: '',
        startsAt: '2017-09-10',
        endsAt: '2017-09-10',
        flag: 'USA',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Challenger',
        title: 'Dreamhack Atlanta',
        region: 'Americas',
        location: 'Atlanta, GA',
        venue: '',
        address: '',
        startsAt: '2017-07-21',
        endsAt: '2017-07-23',
        flag: 'USA',
        website: 'https://usa.dreamhack.com/',
        logo_image: 'dh_atl.png'
      },
      {
        type: 'Challenger',
        title: 'Summer Jam',
        region: 'Americas',
        location: 'Philadelphia, PA',
        venue: '',
        address: '',
        startsAt: '2017-08-11',
        endsAt: '2017-08-13',
        flag: 'USA',
        website: 'http://bigegaming.com/',
        logo_image: 'summer_jam.png'
      },
      {
        type: 'Challenger',
        title: 'Absolute Battle',
        region: 'Americas',
        location: 'Dallas, TX',
        venue: '',
        address: '',
        startsAt: '2017-08-25',
        endsAt: '2017-08-27',
        flag: 'USA',
        website: 'http://absolutebattle.com',
        logo_image: 'ab.png'
      },
      {
        type: 'Challenger',
        title: 'East Coast Throwdown',
        region:'Americas',
        location: 'Morristown, NJ',
        venue: '',
        address: '',
        startsAt: '2017-09-02',
        endsAt: '2017-09-03',
        flag: 'USA',
        website: 'https://ecthrowdown.com/',
        logo_image: 'ect.png'
      },
      {
        type: 'Online',
        title: 'NA West',
        region: 'Americas',
        location: '',
        venue: '',
        address: '',
        startsAt: '2017-08-20',
        endsAt: '2017-08-20',
        flag: 'USA',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Master',
        title: 'The Fall Classic',
        region: 'Americas',
        location: 'Raleigh, NC',
        venue: '',
        address: '',
        startsAt: '2017-09-29',
        endsAt: '2017-10-01',
        flag: 'USA',
        website: 'http://bigegaming.com/',
        logo_image: 'TFC.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Master',
        title: 'TXT',
        region: 'Americas',
        location: 'Santiago, Chile',
        venue: '',
        address: '',
        startsAt: '2017-10-07',
        endsAt: '2017-10-07',
        flag: 'CHL',
        website: 'http://www.tekkenxt.org/',
        logo_image: 'txt.svg',
        pot_bonus: '$10,000'
      },
      {
        type: 'Master',
        title: 'Socal Regionals',
        region: 'Americas',
        location: 'Anaheim, CA',
        venue: '',
        address: '',
        startsAt: '2017-09-22',
        endsAt: '2017-09-24',
        flag: 'USA',
        website: 'http://www.socalregionals.com/',
        logo_image: 'scr.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Regional',
        title: 'Dreamhack Denver',
        region: 'Americas',
        location: 'Denver, CO',
        venue: '',
        address: '',
        startsAt: '2017-10-20',
        endsAt: '2017-10-22',
        flag: 'USA',
        website: 'https://usa.dreamhack.com/',
        logo_image: 'dh_den.png',
        pot_bonus: '$15,000'
      },
      {
        type: 'Challenger',
        title: 'Tattakai Holland',
        region: 'Europe',
        location: 'Amsterdam, Netherland',
        venue: '',
        address: '',
        startsAt: '2017-07-09',
        endsAt: '2017-07-09',
        flag: 'NLD',
        website: 'https://smash.gg/tournament/tattakai-holland-xl/register',
        logo_image: 'th.png'
      },
      {
        type: 'Online',
        title: 'EU East',
        region: 'Europe',
        location: 'Europe',
        venue: '',
        address: '',
        startsAt: '2017-09-10',
        endsAt: '2017-09-10',
        flag: 'EUR',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Challenger',
        title: 'Moscow Fighting Arena',
        region: 'Europe',
        location: 'Moscow, Russia',
        venue: '',
        address: '',
        startsAt: '2017-07-08',
        endsAt: '2017-07-09',
        flag: 'RUS',
        website: 'http://mfa2017.softclub.ru/en',
        logo_image: 'mfa.png'
      },
      {
        type: 'Challenger',
        title: 'Let\'s Play Live Offline Qualifier',
        region: 'Asia-Pacific',
        location: 'Auckland, New Zealand',
        venue: '',
        address: '',
        startsAt: '2017-09-09',
        endsAt: '2017-09-09',
        flag: 'NZL',
        website: 'https://battlefy.com/lets-play-live-lpl/lpl4-tekken7-offline-qualifier/5990e8ecaf93dd0336c9c49b/info?infoTab=details',
        logo_image: 'lpl.png'
      },
      {
        type: 'Master',
        title: 'VSFighting',
        region: 'Europe',
        location: 'Birmingham, UK',
        venue: '',
        address: '',
        startsAt: '2017-08-12',
        endsAt: '2017-08-13',
        flag: 'GBR',
        website: 'https://www.vsfighting.com',
        logo_image: 'VSF.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Challenger',
        title: 'Celtic Throwdown',
        region: 'Europe',
        location: 'Dublin, Ireland',
        venue: '',
        address: '',
        startsAt: '2017-09-02',
        endsAt: '2017-09-03',
        flag: 'IRL',
        website: 'http://celticthrowdown.com',
        logo_image: 'dcct.png'
      },
      {
        type: 'Online',
        title: 'EU West',
        region: 'Europe',
        location: 'Europe',
        venue: '',
        address: '',
        startsAt: '2017-08-27',
        endsAt: '2017-08-27',
        flag: 'EUR',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Master',
        title: 'Tekken Grand Battle',
        region: 'Europe',
        location: 'Lyon, France',
        venue: '',
        address: '',
        startsAt: '2017-09-16',
        endsAt: '2017-09-17',
        flag: 'FRA',
        website: 'http://www.themixup.eu/en',
        logo_image: 'TGB.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Challenger',
        title: 'Milan Games Week',
        region: 'Europe',
        location: 'Milan, Italy',
        venue: '',
        address: '',
        startsAt: '2017-09-29',
        endsAt: '2017-09-30',
        flag: 'EUR',
        website: 'https://pge.gg/tournaments/34',
        logo_image: 'mgw.png'
      },
      {
        type: 'Master',
        title: 'Barcelona Games World',
        region: 'Europe',
        location: 'Barcelona, Spain',
        venue: '',
        address: '',
        startsAt: '2017-10-06',
        endsAt: '2017-10-07',
        flag: 'ESP',
        website: 'http://tekkenbgw.es/',
        logo_image: 'bgw.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Challenger',
        title: 'Warsaw Games Week',
        region: "Europe",
        location: 'Warsaw, Poland',
        venue: '',
        address: '',
        startsAt: '2017-10-13',
        endsAt: '2017-10-15',
        flag: 'POL',
        website: '',
        logo_image: 'wgw.jpg'
      },
      {
        type: 'Master',
        title: 'Dreamolition Derby',
        region: 'Europe',
        location: 'Germany, Munich',
        venue: '',
        address: '',
        startsAt: '2017-09-01',
        endsAt: '2017-09-03',
        flag: 'DEU',
        website: 'http://dreamolitionderby.de/',
        logo_image: 'derby.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Regional',
        title: 'Paris Games Week',
        region: 'Europe',
        location: 'Paris, France',
        venue: '',
        address: '',
        startsAt: '2017-11-04',
        endsAt: '2017-11-05',
        flag: 'FRA',
        website: 'https://smash.gg/tournament/tekken-world-tour-paris-games-week-eu-regional-final-2017',
        logo_image: 'pgw.png',
        pot_bonus: '$15,000'
      },
      {
        type: 'Challenger',
        title: 'Thaigeruppercut',
        region: 'Asia-Pacific',
        location: 'Bangkok, Thailand',
        venue: '',
        address: '',
        startsAt: '2017-07-01',
        endsAt: '2017-07-02',
        flag: 'THA',
        website: 'http://www.tguthailand.com/tournament/',
        logo_image: 'TGU.png'
      },
      {
        type: 'Master',
        title: 'Rev Major',
        region: 'Asia-Pacific',
        location: 'Manila, Philippines',
        venue: '',
        address: '',
        startsAt: '2017-07-08',
        endsAt: '2017-07-09',
        flag: 'PHL',
        website: 'http://revmajor.playbookph.com/',
        logo_image: 'rev.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Online',
        title: 'Asia Pacific - East',
        region: 'Asia-Pacific',
        location: 'Asia-Pacific',
        venue: '',
        address: '',
        startsAt: '2017-09-10',
        endsAt: '2017-09-10',
        flag: '',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Challenger',
        title: 'Abuget Cup',
        region: 'Asia-Pacific',
        location: 'Jakarta, Indonesia',
        venue: '',
        address: '',
        startsAt: '2017-07-29',
        endsAt: '2017-07-30',
        flag: 'IDN',
        website: 'http://www.jakartagamefest.com/events/abuget-cup-2017/',
        logo_image: 'abug.png'
      },
      {
        type: 'Online',
        title: 'Asia Pacific - Southeast',
        region: 'Asia-Pacific',
        location: 'Asia-Pacific',
        venue: '',
        address: '',
        startsAt: '2017-08-20',
        endsAt: '2017-08-20',
        flag: '',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Online',
        title: 'Asia Pacific - Australia',
        region: 'Asia-Pacific',
        location: 'Asia-Pacific',
        venue: '',
        address: '',
        startsAt: '2017-08-13',
        endsAt: '2017-08-13',
        flag: 'AUS',
        website: 'https://smash.gg/twtonline',
        logo_image: ''
      },
      {
        type: 'Challenger',
        title: 'Ozhadou Nationals 15',
        region: 'Asia-Pacific',
        location: 'Sydney, Australia',
        venue: '',
        address: '',
        startsAt: '2017-09-15',
        endsAt: '2017-09-17',
        flag: 'AUS',
        website: 'http://www.ohn.ozhadou.net/',
        logo_image: 'ohn.svg'
      },
      {
        type: 'Master',
        title: 'TWFighter Major',
        region: 'Asia-Pacific',
        location: 'Taipei, Taiwan',
        venue: '',
        address: '',
        startsAt: '2017-10-07',
        endsAt: '2017-10-08',
        flag: 'TWN',
        website: 'http://twfmajor.com/',
        logo_image: 'twf.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Master',
        title: 'Tokyo Tekken Masters',
        region: 'Asia-Pacific',
        location: 'Tokyo, Japan',
        venue: '',
        address: '',
        startsAt: '2017-09-03',
        endsAt: '2017-09-03',
        flag: 'JPN',
        website: 'http://mastercupofficial.com/17391',
        logo_image: 'ttm.png'
      },
      {
        type: 'Master',
        title: 'TEKKEN World Tour Korea',
        region: 'Asia-Pacific',
        location: 'Seoul, Korea',
        venue: '',
        address: '',
        startsAt: '2017-08-06',
        endsAt: '2017-08-06',
        flag: 'KOR',
        website: 'https://beastapac.com/tekken-7-master-tournament/',
        logo_image: 'twt_korea.png',
        pot_bonus: '$10,000'
      },
      {
        type: 'Regional',
        title: 'South East Asia Major',
        region: 'Asia-Pacific',
        location: 'Singapore',
        venue: '',
        address: '',
        startsAt: '2017-10-13',
        endsAt: '2017-10-15',
        flag: 'SGP',
        website: 'http://www.beastapac.com',
        logo_image: 'seam.png',
        pot_bonus: '$15,000'
      },
      {
        type: 'Challenger',
        title: 'The Colosseum',
        region: 'Europe',
        location: 'Rome, Italy',
        venue: '',
        address: '',
        startsAt: '2017-10-14',
        endsAt: '2017-10-15',
        flag: 'ITA',
        website: '',
        logo_image: 'cf.svg'
      },
      {
        type: 'Finals',
        title: 'Tekken World Tour Global Finals',
        region: 'Finals',
        location: 'San Francisco, USA',
        venue: 'Metreon',
        address: '135 4th St #4000, San Francisco, CA',
        startsAt: '2017-11-12',
        endsAt: '2017-11-12',
        flag: 'USA',
        website: 'https://www.eventbrite.com/e/tekken-world-tour-finals-presented-by-nissin-cup-noodles-tickets-38368375845',
        logo_image: '',
        pot_bonus: '$50,000'
      }
    ]);
  }

  render() {
    const query = qs.parse(this.props.location.search.substring(1));
    const { types, regions } = query;

    let typeFilters = [];
    ['Master', 'Challenger', 'Online', 'Regional', 'EVO', 'Finals'].forEach((type, idx) => {
      const typeKey = toSnakeCase(type);
      typeFilters.push(<li key={idx}>
        <ToggleLink
          to={{ search: search(toggle(query, 'types', type)) }}
          filterValue={typeKey}
        >
          <FormattedMessage id={`schedule.filter.type.${typeKey}`} />
        </ToggleLink>
      </li>);
    });

    let regionFilters = [];
    ['Americas', 'Europe', 'Asia-Pacific'].forEach((region, idx) => {
      const regionKey = toSnakeCase(region);
      regionFilters.push(<li key={idx}>
        <ToggleLink
          to={{ search: search(toggle(query, 'regions', region)) }}
          filterValue={regionKey}
        >
          <FormattedMessage id={`schedule.filter.region.${regionKey}`} />
        </ToggleLink>
      </li>);
    });

    const events = this.state.events.filter((event) => {
      if (types && !types.includes(event.type)) return false;
      if (regions && !regions.includes(event.region)) return false;
      return true;
    });

    const pageTitle = this.props.intl.formatMessage({id: 'schedule.header.title'});

    return (
      <div>
        <PageHeader currentPageTitle={pageTitle} banner={banner} />

        <main className="l-content">
          <h2 className="o-visuallyhidden">
            <FormattedMessage id="schedule.filter.title" />
          </h2>

          <div className="o-flex">
            <div className="o-flex__primary o-flex">
              <div className="c-filter-group">
                <h3 className="c-filter-group__title o-h4 o-hidden@small-only t-text-weight--light t-font-family--body">
                  <FormattedMessage id="schedule.filter.type.title" />
                </h3>

                <DropdownSelect title={this.getTypesTitle(query)}>
                  <ul className="o-list-reset c-filter-group__filters">
                    {typeFilters}
                    <li>
                      <ToggleLink to={{ search: search(clear(query, 'types')) }}>
                        <FormattedMessage id="schedule.filter.type.all" />
                      </ToggleLink>
                    </li>
                  </ul>
                </DropdownSelect>
              </div>

              <div className="c-filter-group o-flex__primary">
                <h3 className="c-filter-group__title o-h4 o-hidden@small-only t-text-weight--light t-font-family--body">
                  <FormattedMessage id="schedule.filter.region.title" />
                </h3>

                <DropdownSelect title={this.getRegionsTitle(query)}>
                  <ul className="o-list-reset c-filter-group__filters">
                    {regionFilters}
                    <li>
                      <ToggleLink to={{ search: search(clear(query, 'regions')) }}>
                        <FormattedMessage id="schedule.filter.region.all" />
                      </ToggleLink>
                    </li>
                  </ul>
                </DropdownSelect>
              </div>
            </div>
            <div className="c-filter-group c-filter-group--clear">
              <Link to={{ search: '' }} className="o-button o-button--clear o-h5 t-no-margin">
                <svg className="o-icon o-icon--small o-button__icon" width="16" height="16" role="presentation">
                  <use xlinkHref={`${icons}#icon-close`} />
                </svg>
                <span className="o-button__label">
                  <FormattedMessage id="schedule.filter.clear" />
                </span>
              </Link>
            </div>
          </div>

          <EventList events={events} />
        </main>
      </div>
    );
  }
}

Schedule.propTypes = {
  location: PropTypes.shape({ search: PropTypes.string.isRequired }).isRequired
};

export default injectIntl(Schedule);
