import React, { Component } from 'react';
import { FormattedMessage, injectIntl } from 'react-intl';
import { NavLink } from 'react-router-dom';
import PropTypes from 'prop-types';
import qs from 'qs';

import PageHeader from '../components/PageHeader';
import DropdownSelect from '../components/DropdownSelect';
import StandingList from './standings/StandingList';

import banner from '../images/mockup-schedule-banner.png';

const ALL_REGIONS = ['Americas', 'Europe', 'Asia-Pacific'];
const DEFAULT_REGION = ALL_REGIONS[0];

function search(query) {
  return qs.stringify(query, { encodeValuesOnly: true });
}

function searchToObject(search) {
  var pairs = search.substring(1).split("&"),
    obj = {},
    pair,
    i;

  for ( i in pairs ) {
    if ( pairs[i] === "" ) continue;

    pair = pairs[i].split("=");
    obj[ decodeURIComponent( pair[0] ) ] = decodeURIComponent( pair[1] );
  }

  return obj;
}

function toSnakeCase(str) {
  return str.replace(/[ -]/, '_').toLowerCase();
}

function ToggleLink(props) {
  function isActive(match, location) {
    const values = searchToObject(location.search);

    for (let property in values) {
      if (values.hasOwnProperty(property)
          && values[property] !== undefined
          && toSnakeCase(values[property]) === props.filterValue) {
        return true
      }
    }
    return false
  }

  return (<NavLink {...props} isActive={isActive} activeClassName="is-active" className="o-toggle o-toggle--filter o-h5 c-filter-group__filter" replace />);
}

function updateFilter(filters, type, value) {
  return { ...filters, [type]: value };
}

class Standings extends Component {
  constructor(props) {
    super(props);

    this.state = {
      standings: []
    };
  }

  componentDidMount() {
    this.fetchStandings().then((standings) => {
      this.setState(() => ({ standings }));
    });
  }

  setDefaultRegion() {
    const searchObj = searchToObject(this.props.location.search);
    if (!searchObj.region) {
      const query = qs.parse(this.props.location.search.substring(1));
      this.props.history.push({
        ...this.props.location,
        search: search(updateFilter(query, 'region', DEFAULT_REGION))
      });
    }
  }

  getRegionsTitle(query) {
    if (query && query.regions && query.regions.length === 1) {
      return this.props.intl.formatMessage({id: `schedule.filter.region.${toSnakeCase(query.regions[0])}`})
    }
    return this.props.intl.formatMessage({id: 'schedule.filter.region.title'});
  }

  fetchStandings() {
    // TODO: Replace with actual API call
    const standings = require('../standings/data.json')
      .map((standing) => {
        // Strip -T tied rank notation
        if (typeof standing.Rank === 'string') {
          standing.Tied = standing.Rank.match(/T$/) ? true : false;
          standing.Rank = standing.Rank.replace(/-T$/, '');
        } else {
          standing.Tied = false;
        }
        return standing;
      });
    return Promise.resolve(standings);
  }

  componentDidUpdate() {
    const query = qs.parse(this.props.location.search.substring(1));
    const { region } = query;

    // ideally this would be in component did mount, but clicking nav can get you around this,
    // this probably fits better elsewhere.
    if (!region) {
      this.setDefaultRegion()
    }
  }

  render() {
    const pageTitle = this.props.intl.formatMessage({id: 'standings.header.title'});
    const query = qs.parse(this.props.location.search.substring(1));
    const { characters, region } = query;

    let regionFilters = [];
    ALL_REGIONS.forEach((region, idx) => {
      const regionKey = toSnakeCase(region);
      regionFilters.push(<li key={idx}>
        <ToggleLink
          to={{ search: search(updateFilter(query, 'region', region)) }}
          filterValue={regionKey}
        >
          <FormattedMessage id={`schedule.filter.region.${regionKey}`} />
        </ToggleLink>
      </li>);
    });

    const standings = this.state.standings.filter((standing) => {
      // if (events && !events.includes(standing.event)) return false;
      if (region && !region.includes(standing.Region)) return false;
      if (characters && !characters.includes(standing['Main Character'])) return false;

      return true;
    });

    return (
      <div>
        <PageHeader currentPageTitle={pageTitle} banner={banner} />

        <main className="l-content">
          <h2 className="o-visuallyhidden">
            <FormattedMessage id="standings.filter.title" />
          </h2>

          <div className="o-flex t-margin--bottom">
            <div className="o-flex__primary o-flex">
              <div className="c-filter-group o-flex__primary">
                <h3 className="c-filter-group__title o-h4 o-hidden@small-only t-text-weight--light t-font-family--body">
                  <FormattedMessage id="schedule.filter.region.title" />
                </h3>

                <DropdownSelect title={this.getRegionsTitle(query)}>
                  <ul className="o-list-reset c-filter-group__filters">
                    {regionFilters}
                    {/*<li>
                      <ToggleLink to={{ search: search(clear(query, 'regions')) }}>
                        <FormattedMessage id="schedule.filter.region.all" />
                      </ToggleLink>
                    </li>*/}
                  </ul>
                </DropdownSelect>
              </div>
            </div>
            {/*<div className="c-filter-group c-filter-group--clear">
              <Link to={{ search: '' }} className="o-button o-button--clear o-h5 t-no-margin">
                <svg className="o-icon o-button__icon" width="16" height="16" role="presentation">
                  <use xlinkHref={`${icons}#icon-close`} />
                </svg>
                <span className="o-button__label">
                  <FormattedMessage id="filter.clear" />
                </span>
              </Link>
            </div>*/}
          </div>

          <StandingList standings={standings} />
        </main>
      </div>
    );
  }
}

Standings.propTypes = {
  location: PropTypes.shape({ search: PropTypes.string.isRequired }).isRequired
};

export default injectIntl(Standings);
