import 'whatwg-fetch'
import React, { Component } from 'react';
import { FormattedMessage } from 'react-intl';
import moment from 'moment';

import { TWITCH_CHANNEL_ID, TWITCH_CLIENT_ID } from '../../config';
import Spinner from '../../components/Spinner';

import icons from '../../images/icons.svg';

function VideoItem({video}) {
  const {title, preview, link, date, length} = video;
  var timeFormat =  Math.floor(moment.duration(length,'seconds').asHours()) + ':' + moment.duration(length,'seconds').minutes() + ':' + moment.duration(length,'seconds').seconds();

  return (
    <a href={link} className="o-link o-link--alt o-block t-margin--bottom">
      <div className="o-fixed-ratio t-margin-x-small--bottom">
        <div className="o-fixed-ratio__inner">
          <img src={preview} alt="" />
          <svg className="o-icon o-icon--medium o-absolute-center" width="16" height="16" role="presentation">
            <title>
              <FormattedMessage id="home.video.play_button" />
            </title>
            <use xlinkHref={`${icons}#icon-play`} />
          </svg>
        </div>
      </div>

      <h3 className="o-h3 t-no-margin o-truncate-with-ellipsis">{title}</h3>
      <h4 className="o-h4 o-flex t-no-margin t-color--grey t-text-weight--light t-font-family--body">
        <span className="o-flex__primary">{moment(date).fromNow()}</span>
        <span>{timeFormat}</span>
      </h4>
    </a>
  );
}

class HomeVideo extends Component {
  constructor(props) {
    super(props);
    this.state = {
      videos: [],
      limit: 32,
      displayCount: 8,
      offset: 0,
    };
  }

  componentDidMount() {
    const { limit, offset } = this.state;
    this.fetchVideos(limit, offset);
  }

  fetchWithTwitchHeaders(url) {
    return fetch(url, {
      headers: {
        Accept: 'application/vnd.twitchtv.v5+json',
        'Client-ID': TWITCH_CLIENT_ID,
      }
    });
  }

  fetchVideos(limit, offset) {
    let twitchApi = 'https://api.twitch.tv/kraken';
    return this.fetchWithTwitchHeaders(`${twitchApi}/channels/${TWITCH_CHANNEL_ID}/videos?limit=${this.state.limit}`)
      .then((response) => response.json())
      .then((data) => this.setState({ videos: data.videos.map(this.parseVideo)}));
  }

  parseVideo(video) {
    return {
      id: video.broadcast_id,
      title: video.title,
      preview: video.preview.large,
      views: video.views,
      link: video.url,
      date: video.published_at,
      length: video.length
    };
  }

  loadMore() {
    return this.setState({ displayCount: this.state.displayCount + 8 });
  }

  render() {
    const items = this.state.videos.slice(0, this.state.displayCount).map((video, idx) => {
      return(
        <li key={idx} className="l-grid__item l-width--6-of-12 l-width--3-of-12@medium">
          <VideoItem video={video}/>
        </li>
      );
    });

    return (
      <section id="video" className="l-content t-margin--bottom">
        <h2 className="o-h3 o-bordered t-color--grey t-text-transform--uppercase">
          <FormattedMessage id="home.video.title" />
        </h2>
        <ul className="l-grid l-grid--narrow-gutter">
          { items }

         {this.state.videos.length > this.state.displayCount &&
          <li className="l-grid__item l-width--12-of-12">
            <button
              className="o-button o-button--load t-text-transform--uppercase"
              onClick={this.loadMore}
            >
              <span className="o-link__label t-margin-x-small--right">Load more</span>
              <Spinner isWaiting={false} />
            </button>
          </li>
        }
        </ul>
      </section>
    );
  }
}

export default HomeVideo;
