import React from 'react';
import PropTypes from 'prop-types';
import { FormattedMessage, injectIntl } from 'react-intl';

import Flag from '../../components/Flag';
import noEvent from '../../images/no_event.svg';


function EventLink(event) {
  if (event.website) {
    return (
      <a href={event.website} className="o-button o-button--reversed o-h3 t-no-margin">
        <span className="o-button__label">
          <FormattedMessage id="schedule.event.button" />
        </span>
      </a>
    );
  }
  return null
}

EventLink.propTypes = {
  website: PropTypes.string
};


function EventDates(event) {
  if (event.startsAt === '2017')
    return 'TBA';

  const startsAt = new Date(event.startsAt).getUTCDate();
  const endsAt = new Date(event.endsAt).getUTCDate();

  if (startsAt !== endsAt) {
    return startsAt + '–' + endsAt
  }
  return startsAt
}

EventDates.propTypes = {
  startsAt: PropTypes.string.isRequired,
  endsAt: PropTypes.string
}


function getImageUrl(logoImage, suffix) {
  const splitLogoImage = logoImage.split('.');

  return require(`../../images/events/${splitLogoImage.slice(0, splitLogoImage.length - 1)}${suffix}.${splitLogoImage[splitLogoImage.length - 1]}`);
}

function EventLogo(event) {
  if (event.logo_image) {
    try {
      const image300w = getImageUrl(event.logo_image, '_300w');
      const image150w = getImageUrl(event.logo_image, '_150w');

      return (
        <img
          srcSet={`${image300w} 300w, ${image150w} 150w`}
          sizes="(min-width: 66rem) 160px, 20vw"
          alt={event.title}
        />
      );
    } catch(error) {
      const imageSrc = require(`../../images/events/${event.logo_image}`);
      return (<img src={imageSrc} alt={event.title} />);
    }
  }

  const fallbackSrc = require('../../images/no_event.svg');
  return (<img src={fallbackSrc} alt={event.title} />)
}

EventLogo.propTypes = {
  logo_image: PropTypes.string
}


function EventListItem({ event }) {
  return (
    <li className="c-event">
      <p className="c-event__summary o-h3">
        <span className="o-block o-visuallyhidden@small-only t-text-weight--normal">{event.region}</span>
        <span className="o-block o-h1 t-no-margin t-color--text t-text-weight--normal">{ EventDates(event) }</span>
        <span className="o-block t-text-weight--light">{event.location}</span>
        <Flag country={event.flag} className="c-event__flag o-flag--large" />
      </p>
      <div className="c-event__logo">{ EventLogo(event) }</div>
      <div className="c-event__details">
        <h3 className="c-event__title o-h4">
          <span className="c-event__name o-h1 t-no-margin t-text-transform--uppercase">{event.title}</span>
          <span className={`c-event__type o-event-type o-event-type--${event.type}`}>
            {event.type}
            {event.pot_bonus !== undefined ? <span className="c-event__pot-bonus">+{event.pot_bonus}</span> : ''}
          </span>
        </h3>
        <div className="c-event__extra-details">
          <div className="c-event__button">
            { EventLink(event) }
          </div>
          <p className="o-h4 c-event__address">
            {event.venue}
            <br />
            {event.address}
          </p>
        </div>
      </div>
    </li>
  );
}

const eventShape = PropTypes.shape({
  region: PropTypes.oneOf(['Americas', 'Europe', 'Asia-Pacific', 'Finals']).isRequired,
  type: PropTypes.oneOf(['Master', 'Challenger', 'Online', 'Regional', 'EVO', 'Finals']).isRequired,
  title: PropTypes.string.isRequired,
  location: PropTypes.string.isRequired,
  venue: PropTypes.string.isRequired,
  address: PropTypes.string.isRequired,
  startsAt: PropTypes.string.isRequired,
  endsAt: PropTypes.string.isRequired,
  flag: PropTypes.string.isRequired,
  website: PropTypes.string.isRequired
});

EventListItem.propTypes = {
  event: eventShape.isRequired
};

function EventList({ events, intl }) {
  const monthNames = [
    intl.formatMessage({id: 'month.january'}),
    intl.formatMessage({id: 'month.february'}),
    intl.formatMessage({id: 'month.march'}),
    intl.formatMessage({id: 'month.april'}),
    intl.formatMessage({id: 'month.may'}),
    intl.formatMessage({id: 'month.june'}),
    intl.formatMessage({id: 'month.july'}),
    intl.formatMessage({id: 'month.august'}),
    intl.formatMessage({id: 'month.september'}),
    intl.formatMessage({id: 'month.october'}),
    intl.formatMessage({id: 'month.november'}),
    intl.formatMessage({id: 'month.december'}),
  ];

  const sorted = events.sort((a, b) => {
    function start(event) { return new Date(event.startsAt); };
    return start(a) < start(b) ? -1 : 1;
  });

  const grouped = sorted.reduce((groups, event) => {
    // Making sure events w/o a date are at the bottom of the list
    const [year, month] = event.startsAt.match(/-/) ?
      event.startsAt.split('-') :
      [event.startsAt, 99];
    const key = [year, month].join('-');
    const prev = groups[key] || [];
    const group = [...prev, event];
    return { ...groups, [key]: group };
  }, {});

  const noResults = events.length === 0
    ? <img alt="No Events" src={noEvent} className="c-event-list c-event-list--empty"/>
    : undefined

  const sections = Object.keys(grouped).sort().map((key, index) => {
    const group = grouped[key];

    const items = group.map((event, idx) => {
      return (<EventListItem key={idx} event={event} />);
    });
    const monthName = monthNames[new Date(key).getUTCMonth()];
    const monthDisplay = key !== '2017-' ? <h2 className="o-h3 c-event-month t-color--grey t-no-margin t-text-transform--uppercase t-font-family--body">{monthName}</h2> : null;

    return (
      <li key={index}>
        { monthDisplay }
        <ul className="o-list-reset">{items}</ul>
      </li>
    );
  });

  return (
    <div>
      <ul className="c-event-list">
        {sections}
      </ul>
      {noResults}
    </div>);
}

EventList.propTypes = {
  events: PropTypes.arrayOf(eventShape).isRequired
};

export default injectIntl(EventList);
