import React from 'react';
import PropTypes from 'prop-types';

import Flag from '../../components/Flag';

const standingShape = PropTypes.shape({
  Rank: PropTypes.any.isRequired,
  Player: PropTypes.string.isRequired,
  Points: PropTypes.number.isRequired,
  "Full Name": PropTypes.string.isRequired,
  Country: PropTypes.string,
  "Main Character": PropTypes.string,
  Region: PropTypes.oneOf(['Americas', 'Europe', 'Asia-Pacific']).isRequired
});

const highlightedItemsCount = 3;

function toSnakeCase(str) {
  return str.replace(/[ -]/, '_').toLowerCase();
}

function CharacterImage(props) {
  try {
    const characterImage = toSnakeCase(props.characterName);
    const image600w = require(`../../images/characters/${characterImage}.png`);
    const image300w = require(`../../images/characters/${characterImage}.png`);
    const image150w = require(`../../images/characters/${characterImage}.png`);

    return (
      <img
        srcSet={`${image600w} 600w, ${image300w} 300w, ${image150w} 150w`}
        sizes={props.sizes}
        alt={props.characterName}
      />
    );
  } catch(error) {
    const fallbackSrc = require('../../images/no_character.svg');
    return (<img src={fallbackSrc} alt={props.characterName} />)
  }
}

CharacterImage.propTypes = {
  characterName: PropTypes.string,
  sizes: PropTypes.string.isRequired
}


function StandingListItem({ standing }) {
  // TODO move to styles, __flag-us?
  let flagBg;
  try {
    flagBg = require(`../../images/flags/${standing.Country.toLowerCase()}.png`) ;
  } catch(err) {
    flagBg = null;
    
    // missing flag list
    // if(!window.flags){
    //   window.flags = new Set();
    // }
    // window.flags.add(standing.Country);
    // console.log('missing flag', window.flags.values());
  }

  const styles = standing.Country ? {
    backgroundImage:
      'linear-gradient(90deg, transparent, #131313),' +
      'url(' + flagBg + ')'
    } : {};
  return (
    <li className="c-standing" style={styles}>
      <span className="c-standing__position o-h1">
        {standing.Rank}
        {standing.Tied ? <span className="c-standing__tied">tied</span> : ''}
      </span>
      <span>
        <Flag country={standing.Country} className="c-standing__flag o-flag--large o-flag--medium@medium" />
      </span>
      <div className="c-standing__username">
        <h3 className="o-h3 t-no-margin o-truncate-with-ellipsis">{standing.Player}</h3>
        <span className="c-standing__name o-h4 t-font-family--body t-text-weight--normal o-hidden@medium">{standing['Full Name']}</span>
      </div>
      <span className="c-standing__name o-h4 t-font-family--body t-text-weight--normal o-hidden@small-only">{standing['Full Name']}</span>
      <div>
        <div className="c-standing__points o-h2 t-no-margin">
          {standing.Points}
        </div>
        <div className="c-standing__character o-h4 t-font-family--body t-text-weight--normal o-hidden@medium t-no-margin">{standing['Main Character']}</div>
      </div>
      <span className="c-standing__character o-h4 t-font-family--body t-text-weight--normal o-hidden@small-only">{standing['Main Character']}</span>
      <span className="c-standing__image">
        <CharacterImage
          characterName={standing['Main Character']}
          sizes="(min-width: 66rem) 5rem, 4rem"
        />
      </span>
    </li>
  );
}

StandingListItem.propTypes = {
  standing: standingShape.isRequired
};


function StandingHighlightedItem({ standing }) {
  let flagBg;
  try {
    flagBg = require(`../../images/flags/${standing.Country.toLowerCase()}.png`) ;
  } catch(err) {
    flagBg = null;
  }

  const styles = standing.Country
    ? { backgroundImage: 'url(' + flagBg + ')' }
    : {};
  return (
    <li className="l-grid__item l-width--4-of-12@medium">
      <div className="c-standing c-standing--highlight" style={styles}>
        <div className="c-standing__image">
          <CharacterImage
            characterName={standing['Main Character']}
            sizes="(max-width: 50rem) 40vw, 8.8rem"
          />
        </div>

        <div className="c-standing__top">
          <span className="c-standing__position o-h1">{standing.Rank}</span>
          <span className="c-standing__points o-h2">{standing.Points}</span>
          <span className="c-standing__character o-h4 t-font-family--body t-text-weight--normal">{standing['Main Character']}</span>
        </div>

        <h3 className="o-h5 t-no-margin c-standing__bottom">
          <span className="c-standing__names">
            <span className="c-standing__username o-h3">{standing.Player}</span>
            <span className="c-standing__name o-h4 t-font-family--body t-text-weight--normal">{standing['Full Name']}</span>
          </span>
        </h3>
      </div>
    </li>
  );
}

StandingHighlightedItem.propTypes = {
  standing: standingShape.isRequired
};

function StandingList({ standings }) {
  // TODO: How will the top three behave when filtered? Top three out of filtered standings?

  const highlightedItems = standings.slice(0, highlightedItemsCount).map((standing, idx) => {
    return (<StandingHighlightedItem key={idx} standing={standing} />);
  });;

  const listItems = standings.slice(highlightedItemsCount, standings.length).map((standing, idx) => {
    return (<StandingListItem key={idx} standing={standing} />);
  });

  return (
    <div>
      <ul className="l-grid">{ highlightedItems }</ul>
      <ul className="o-list-reset">{ listItems }</ul>
    </div>
  );
}

StandingList.propTypes = {
  standings: PropTypes.arrayOf(standingShape).isRequired
};

export default StandingList;
