provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-aws"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-aws/"
}

variable "service_name" { default = "twt" }
variable "service_env" { default = "production" }

data "terraform_remote_state" "aws" {
  backend = "s3"
  config {
    bucket = "twitch-esports-aws"
    key = "tfstate/esports/terraform/aws/raw"
    profile = "twitch-esports-aws"
    region = "us-east-1"
  }
}

resource "aws_s3_bucket" "default" {
  bucket = "${var.service_name}-${var.service_env}"
  website {
    index_document = "index.html"
    error_document = "error.html"
  }
  policy = <<EOF
{
  "Version":"2012-10-17",
  "Statement":[
    {
      "Sid":"AddPerm",
      "Effect":"Allow",
      "Principal": "*",
      "Action":["s3:GetObject"],
      "Resource":["arn:aws:s3:::${var.service_name}-${var.service_env}/*"]
    }
  ]
}
EOF
}

resource "aws_ecr_repository" "frontend" {
  name = "${var.service_name}-frontend"
}

resource "aws_iam_role" "task" {
  name = "ecs_${var.service_name}_task_role"
  assume_role_policy = <<EOF
{
  "Version": "2008-10-17",
  "Statement": [{
    "Action": "sts:AssumeRole",
    "Principal": {
      "Service": [
        "ecs-tasks.amazonaws.com"
      ]
    },
    "Effect": "Allow"
  }]
}
EOF
}

resource "aws_iam_role_policy" "task_s3" {
  name = "ecs_${var.service_name}_task_role_s3"
  role = "${aws_iam_role.task.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": ["s3:*"],
      "Resource": ["${aws_s3_bucket.default.arn}"]
    },
    {
      "Effect": "Allow",
      "Action": ["s3:*"],
      "Resource": ["${aws_s3_bucket.default.arn}/*"]
    }
  ]
}
EOF
}

resource "template_file" "default" {
  template = "${file("task-definitions/twt.json")}"
  vars {
    name = "${var.service_name}"
    region = "us-west-2"
    env = "${var.service_env}"
    image_frontend = "${replace(aws_ecr_repository.frontend.repository_url, "https://", "")}:latest"
    port_frontend = 80
    cpu = 256
    mem = 256
  }
}

resource "aws_ecs_task_definition" "default" {
  family = "${var.service_name}"
  container_definitions = "${template_file.default.rendered}"
  task_role_arn = "${aws_iam_role.task.arn}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-05-19"
  name = "${var.service_name}"
  prefix = "${replace(var.service_name, "-", "_")}"
  alb = true
  nginx = true
  container_port = "80"
  desired_count = 4
  env = "production"
  certificate_arn = "arn:aws:acm:us-west-2:954268612818:certificate/bc6ef236-5ba0-4fd8-abe5-1eb774bdb559"
  task_definition = "${aws_ecs_task_definition.default.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "vpc_id")}"
  cluster = "programs"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.aws.alarm_sns_topic_arn}"
}

output "ecr_frontend_url" {
  value = "${aws_ecr_repository.frontend.repository_url}"
}

resource "aws_route53_record" "production" {
  zone_id = "Z134ZP7BZBBDAY" # tekkenworldtour.com
  name = ""
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

resource "aws_route53_record" "production_www" {
  zone_id = "Z134ZP7BZBBDAY" # tekkenworldtour.com
  name = "www"
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}
