package backend

import (
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/acm"
	"github.com/aws/aws-sdk-go/service/acmpca"
)

// Client contains methods for AWS interactions to allow easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	ACMRequestCertificate(input *acm.RequestCertificateInput) (*acm.RequestCertificateOutput, error)
	ACMListCertificates(input *acm.ListCertificatesInput) (*acm.ListCertificatesOutput, error)
	ACMDescribeCertificate(input *acm.DescribeCertificateInput) (*acm.DescribeCertificateOutput, error)
	ACMExportCertificate(input *acm.ExportCertificateInput) (*acm.ExportCertificateOutput, error)
	ACMPCAGetCertificateAuthorityCertificate(input *acmpca.GetCertificateAuthorityCertificateInput) (*acmpca.GetCertificateAuthorityCertificateOutput, error)
}

type client struct {
	acm    *acm.ACM
	acmpca *acmpca.ACMPCA
}

// New generates all the AWS clients required for this service using standard credentials
func New(sess *session.Session) Client {
	return &client{
		acm:    acm.New(sess),
		acmpca: acmpca.New(sess),
	}
}

// ACMRequestCertificate calls the underlying RequestCertificate from the acm backend
func (c *client) ACMRequestCertificate(input *acm.RequestCertificateInput) (*acm.RequestCertificateOutput, error) {
	return c.acm.RequestCertificate(input)
}

// ACMListCertificates calls the underlying ListCertificates from the acm backend
func (c *client) ACMListCertificates(input *acm.ListCertificatesInput) (*acm.ListCertificatesOutput, error) {
	return c.acm.ListCertificates(input)
}

// ACMDescribeCertificate calls the underlying DescribeCertificate from the acm backend
func (c *client) ACMDescribeCertificate(input *acm.DescribeCertificateInput) (*acm.DescribeCertificateOutput, error) {
	return c.acm.DescribeCertificate(input)
}

// ACMExportCertificate calls the underlying ExportCertificate from the acm backend
func (c *client) ACMExportCertificate(input *acm.ExportCertificateInput) (*acm.ExportCertificateOutput, error) {
	return c.acm.ExportCertificate(input)
}

// ACMPCAGetCertificateAuthorityCertificate calls the underlying GetCertificateAuthorityCertificate from the acmpca backend
func (c *client) ACMPCAGetCertificateAuthorityCertificate(input *acmpca.GetCertificateAuthorityCertificateInput) (*acmpca.GetCertificateAuthorityCertificateOutput, error) {
	return c.acmpca.GetCertificateAuthorityCertificate(input)
}
