package main

import (
	"encoding/json"
	"flag"
	"log"
	"os"

	"github.com/aws/aws-sdk-go/aws"

	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ec2"
)

// Config represents the AWS parameters determined by terraform
type Config struct {
	AWSRegion         string                 `json:"aws_region"`
	RunInstanceConfig *ec2.RunInstancesInput `json:"run_instance_config"`
}

// MustReadJSONFile reads the file at the given path and unmarshals it into v.
func MustReadJSONFile(path string, v interface{}) {
	file, err := os.Open(path)
	if err != nil {
		log.Fatalf("couldn't open json file %v: %v", path, err)
	}
	decoder := json.NewDecoder(file)
	decoder.DisallowUnknownFields()
	if err := decoder.Decode(v); err != nil {
		log.Fatalf("couldn't decode json file %v: %v", path, err)
	}
}

func main() {
	instanceTypesPath := flag.String("instance-types", "instance-types.json", "file with instance types")
	configPath := flag.String("config", "config.json", "config json file")
	dryRun := flag.Bool("dry-run", false, "don't actually launch instances")

	flag.Parse()

	var instanceTypes []string
	MustReadJSONFile(*instanceTypesPath, &instanceTypes)

	var config Config
	MustReadJSONFile(*configPath, &config)

	config.RunInstanceConfig.MinCount = aws.Int64(1)
	config.RunInstanceConfig.MaxCount = aws.Int64(1)

	sess := session.Must(session.NewSession(&aws.Config{
		Region: aws.String(config.AWSRegion),
	}))
	svc := ec2.New(sess)

	for _, instanceType := range instanceTypes {
		runInstanceConfig := *config.RunInstanceConfig
		runInstanceConfig.InstanceType = aws.String(instanceType)

		if *dryRun {
			log.Printf("would run RunInstances with config: %v", runInstanceConfig)
			continue
		}

		_, err := svc.RunInstances(&runInstanceConfig)

		if err != nil {
			log.Printf("Couldn't launch instance with type %v: %v", instanceType, err)
			continue
		}

		log.Printf("Launched instance with type %v", instanceType)
	}
}
