import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';

export interface CarrotAnalyticsJumpboxProps {
  keyPairName: string
}

export class CarrotAnalyticsJumpbox extends cdk.Construct {
  Instance: ec2.Instance

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CarrotAnalyticsJumpboxProps) {
    super(scope, id);

    const iamRole = new iam.Role(this, 'CarrotAnalyticsJumpboxRole', {
      assumedBy: new iam.ServicePrincipal("ec2.amazonaws.com"),
      managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonEC2RoleforSSM")]
    })

    this.Instance = new ec2.Instance(this, 'CarrotAnalyticsJumpbox', {
      vpc: vpc,
      machineImage: ec2.MachineImage.latestAmazonLinux(),
      instanceType: new ec2.InstanceType("t3.medium"),
      role: iamRole,
      keyName: props.keyPairName,
      blockDevices: [
        {
          deviceName: '/dev/xvda',
          mappingEnabled: true,
          volume: ec2.BlockDeviceVolume.ebs(500, { volumeType: ec2.EbsDeviceVolumeType.GP2 })
        }
      ],
    })

    const jumpBoxSecurityGroup = new ec2.SecurityGroup(this, 'CarrotAnalyticsJumpboxSecurityGroup', {
      vpc: vpc,
      allowAllOutbound: true,
      securityGroupName: 'CarrotAnalyticsJumpboxSG',
      description: 'Access rules for the jumpbox',
    })

    this.Instance.addSecurityGroup(jumpBoxSecurityGroup)
  }
}
