import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ddb from '@aws-cdk/aws-dynamodb';
import * as s3 from '@aws-cdk/aws-s3';
import * as sqs from '@aws-cdk/aws-sqs';
import * as iam from '@aws-cdk/aws-iam';
import * as lambdaEvents from '@aws-cdk/aws-lambda-event-sources'
import lambda = require('@aws-cdk/aws-lambda');

export interface CarrotAnalyticsProps {
  redshiftAddr: string
  redshiftUser: string
  redshiftDatabase: string
  redshitPassParameterArn: string
  invokeFunctionRoles: string[]
}

export class CarrotAnalytics extends cdk.Construct {
  QueriesTable: ddb.Table
  ResultsBucket: s3.Bucket
  ControlLambda: lambda.Function
  ExecutorLambda: lambda.Function
  ExecutorQueue: sqs.Queue
  ExecutorDLQ: sqs.Queue

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CarrotAnalyticsProps) {
    super(scope, id);

    // Queries Table
    this.QueriesTable = new ddb.Table(this, 'CarrotAnalyticsQueries', {
      partitionKey: { name: 'query_id', type: ddb.AttributeType.STRING },
      billingMode: ddb.BillingMode.PAY_PER_REQUEST,
      timeToLiveAttribute: 'ttl',
    })

    this.QueriesTable.addGlobalSecondaryIndex({
      indexName: 'queries-requested_by',
      partitionKey: {
        name: 'requested_by',
        type: ddb.AttributeType.STRING,
      },
      sortKey: {
        name: 'requested_at',
        type: ddb.AttributeType.STRING,
      },
      projectionType: ddb.ProjectionType.ALL,
    })

    this.ResultsBucket = new s3.Bucket(this, 'CarrotAnalyticsResultsBucket', {
      encryption: s3.BucketEncryption.KMS_MANAGED,
      lifecycleRules: [{
        expiration: cdk.Duration.days(7), // This corresponds to the dynamo TTL so make sure that's changed too if you want to change this
      }]
    })

    // We should add a lambda to listen to these and mark them as failed
    this.ExecutorDLQ = new sqs.Queue(this, 'CarrotAnalyticsExecutorDLQ', {})

    this.ExecutorQueue = new sqs.Queue(this, 'CarrotAnalyticsExecutorQueue', {
      deadLetterQueue: {
        maxReceiveCount: 3,
        queue: this.ExecutorDLQ,
      },
      visibilityTimeout: cdk.Duration.minutes(10), // This matches the lambda timeout
    })

    const allowDDBAccess = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "dynamodb:Query",
        "dynamodb:Scan",
        "dynamodb:GetItem",
        "dynamodb:PutItem",
        "dynamodb:UpdateItem",
        "dynamodb:DeleteItem",
      ],
      resources: [
        this.QueriesTable.tableArn,
        this.QueriesTable.tableArn + '/*',
      ]
    })

    const allowSQSAccess = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "sqs:SendMessage",
      ],
      resources: [
        this.ExecutorQueue.queueArn,
      ]
    })

    this.ControlLambda = new lambda.Function(this, 'CarrotAnalyticsControl', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(10),
      handler: 'bin/linux/carrot-analytics-control',
      code: lambda.Code.asset('../bin/carrot-analytics-control.zip'),
      environment: {
        "queriesTableName": this.QueriesTable.tableName,
        "executorSQSQueueURL": this.ExecutorQueue.queueUrl,
        "resultsBucketName": this.ResultsBucket.bucketName,
      }
    })

    this.ControlLambda.addToRolePolicy(allowDDBAccess);
    this.ControlLambda.addToRolePolicy(allowSQSAccess);

    // Allow this to be called by whatever we've specified
    for (let roleArn of props.invokeFunctionRoles) {
      this.ControlLambda.grantInvoke(new iam.ArnPrincipal(roleArn))
    }

    // Allow the Controller to retrieve the results from S3
    this.ControlLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:GetObject",
      ],
      resources: [
        this.ResultsBucket.arnForObjects("*"),
      ]
    }))

    this.ExecutorLambda = new lambda.Function(this, 'CarrotAnalyticsExecutor', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.minutes(10), // No query should take more than 10 minutes.... currently
      handler: 'bin/linux/carrot-analytics-executor',
      code: lambda.Code.asset('../bin/carrot-analytics-executor.zip'),
      environment: {
        "redshiftAddr": props.redshiftAddr,
        "redshiftUser": props.redshiftUser,
        "redshiftDatabase": props.redshiftDatabase, // Password is stored in SSM and is added OOB
        "queriesTableName": this.QueriesTable.tableName,
        "resultsBucketName": this.ResultsBucket.bucketName,
      },
      reservedConcurrentExecutions: 10, // This is essentially going to be the max number of queries running on the cluster at the same time
    })

    // Executor needs to update the status of the query on completion
    this.ExecutorLambda.addToRolePolicy(allowDDBAccess);

    // Allow Executor to retrieve the redshift password from SSM
    this.ExecutorLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "ssm:GetParameter",
        "ssm:GetParameters",
      ],
      resources: [
        props.redshitPassParameterArn,
      ]
    }))

    // Allow the Executor to write the results to S3
    this.ExecutorLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:PutObject",
      ],
      resources: [
        this.ResultsBucket.arnForObjects("*"),
      ]
    }))

    // Link it up to the SQS queue
    this.ExecutorLambda.addEventSource(new lambdaEvents.SqsEventSource(this.ExecutorQueue, {
      batchSize: 1,
    }))
  }
}
