package main

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"time"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	control "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	"code.justin.tv/jleroux/go-isengard2/service"
	"code.justin.tv/jleroux/go-isengard2/service/isengard"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/golang/protobuf/ptypes"
)

var (
	accountID    = "291428775132" //starfruit-support+analytics-prod@amazon.com
	region       = "us-west-2"
	isengardRole = "Admin"
	lambdaARN    = "arn:aws:lambda:us-west-2:291428775132:function:carrot-analytics-prod-CarrotAnalyticsCarrotAnalyti-7F0OU9CXIRNA"
)

func main() {
	sess := getIsengardSession()

	lambdaClient := lambda.New(sess)
	lambdaTransport := twirpLambda.NewClient(lambdaClient, lambdaARN)

	client := control.NewCarrotAnalyticsControlProtobufClient("https://would.you.like.a.carrot.twitch.tv", lambdaTransport)

	//doEnqueuePlaySessionInfo(client)
	doRetrievePlaySessionInfo(client)
}

func doRetrievePlaySessionInfo(client control.CarrotAnalyticsControl) {
	resp, err := client.GetQueryResult(context.Background(), &control.GetQueryResultRequest{
		QueryId: "67353c99-4eb0-4d53-8e88-13ffe8343798",
	})

	if err != nil {
		panic(err)
	}

	bytes, _ := json.MarshalIndent(resp, "", "  ")
	fmt.Println(string(bytes))
}

func doEnqueuePlaySessionInfo(client control.CarrotAnalyticsControl) {
	startDate, err := time.Parse(time.RFC3339, "2020-03-24T00:00:00Z")
	if err != nil {
		panic(err)
	}

	endDate, err := time.Parse(time.RFC3339, "2020-03-27T23:59:00Z")
	if err != nil {
		panic(err)
	}

	playSessionID := "f970108b760743adadb3527ddbbfbb85"

	pStart, err := ptypes.TimestampProto(startDate)
	if err != nil {
		panic(err)
	}

	pEnd, err := ptypes.TimestampProto(endDate)
	if err != nil {
		panic(err)
	}

	// Test your shit
	resp, err := client.EnqueueQuery(context.Background(), &control.EnqueueQueryRequest{
		RequestedBy: "localdev",
		Query: &control.Query{
			StartDate: pStart,
			EndDate:   pEnd,
			QueryParams: &control.Query_PlaySessionInfoParams{
				PlaySessionInfoParams: &control.PlaySessionInfoParameters{
					PlaySessionId: playSessionID,
				},
			},
		},
	})

	if err != nil {
		panic(err)
	}

	fmt.Println(resp.QueryId)
}

func getIsengardSession() *session.Session {
	// Create a new isengard client to establish who the user is and generate credentials necessary to contact services
	svc, err := service.NewIsengardService()
	if err != nil {
		fmt.Println("Failed to initiate isengard client (you probably need to run mwinit)")
		os.Exit(1)
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region),
		Credentials: credentials.NewCredentials(&isengardCredentialsProvider{
			svc:       svc,
			accountID: accountID,
			role:      isengardRole,
		}),
	})

	if err != nil {
		fmt.Printf("Failed to generate credentials for StreamIP: %v\n", err)
		os.Exit(1)
	}

	return sess
}

type isengardCredentialsProvider struct {
	svc        *isengard.Isengard
	accountID  string
	role       string
	expiration int64
}

func (icp *isengardCredentialsProvider) Retrieve() (credentials.Value, error) {
	output, err := icp.svc.GetAssumeRoleCredentials(&isengard.GetAssumeRoleCredentialsInput{
		AWSAccountID: aws.String(icp.accountID),
		IAMRoleName:  aws.String(icp.role),
	})

	if err != nil {
		return credentials.Value{}, err
	}

	result := &service.AssumeRoleResult{}
	err = json.Unmarshal([]byte(*output.AssumeRoleResult), result)
	if err != nil {
		return credentials.Value{}, err
	}

	icp.expiration = result.Credentials.Expiration

	return credentials.Value{
		AccessKeyID:     result.Credentials.AccessKeyID,
		SecretAccessKey: result.Credentials.SecretAccessKey,
		SessionToken:    result.Credentials.SessionToken,
		ProviderName:    "isengardCredentialsProvider",
	}, nil
}

func (icp *isengardCredentialsProvider) IsExpired() bool {
	return time.Now().Unix() >= icp.expiration
}
