package main

import (
	"context"
	"fmt"
	"os"

	control "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	"code.justin.tv/event-engineering/carrot-analytics/executor/svc"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/gobuffalo/packr/v2"
	"github.com/golang/protobuf/jsonpb"
	"github.com/sirupsen/logrus"
)

var executorSvc svc.Client

func init() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	redshiftAddr := os.Getenv("redshiftAddr")
	redshiftUser := os.Getenv("redshiftUser")
	redshiftDatabase := os.Getenv("redshiftDatabase")
	resultsBucketName := os.Getenv("resultsBucketName")
	queriesTableName := os.Getenv("queriesTableName")

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		panic(err)
	}

	ssmClient := ssm.New(sess)

	redshiftPassParamName := "/carrot-analytics/redshiftPass"

	param, err := ssmClient.GetParameter(&ssm.GetParameterInput{
		Name:           &redshiftPassParamName,
		WithDecryption: aws.Bool(true),
	})

	if err != nil {
		panic(err)
	}

	redshiftPass := *param.Parameter.Value

	// We're going to package up the query templates with packr
	queries := packr.New("queries", "../../../queries")

	// Create the Service
	executorSvc = svc.New(sess, redshiftAddr, redshiftUser, redshiftPass, redshiftDatabase, resultsBucketName, queriesTableName, queries, logger)
}

func main() {
	lambda.Start(handler)
}

func handler(ctx context.Context, sqsEvent events.SQSEvent) error {
	// There should only be 1 of these as batchSize is set to 1
	if len(sqsEvent.Records) > 1 {
		return fmt.Errorf("Expected 1 record, got %v", len(sqsEvent.Records))
	}

	var queryID events.SQSMessageAttribute
	var resultPath events.SQSMessageAttribute
	var ok bool

	for _, message := range sqsEvent.Records {
		if queryID, ok = message.MessageAttributes["query_id"]; !ok {
			return fmt.Errorf("Missing attribute value query_id")
		}

		if resultPath, ok = message.MessageAttributes["result_path"]; !ok {
			return fmt.Errorf("Missing attribute value result_path")
		}

		var query control.Query

		err := jsonpb.UnmarshalString(message.Body, &query)

		if err != nil {
			return err
		}

		err = executorSvc.ExecuteQuery(ctx, *queryID.StringValue, *resultPath.StringValue, &query)

		if err != nil {
			return err
		}
	}

	return nil
}
