package svc

import (
	"bytes"
	"context"
	"database/sql"
	"errors"
	"regexp"
	"text/template"
	"time"

	control "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	"github.com/golang/protobuf/ptypes"
)

type broadcastSessionInfoTemplateVars struct {
	IngestSessionID string
	StartDateTime   string
	EndDateTime     string
	StartDate       string
	EndDate         string
}

var validIngestSessionID = regexp.MustCompile("^[0-9a-fA-F-]+$")

func (c *client) formatBroadcastInfoQuery(ctx context.Context, db *sql.DB, startDateTime, endDateTime, startDate, endDate time.Time, params *control.BroadcastInfoParameters) (*sql.Rows, error) {
	// Grab the SQL from packr and add the params in
	query, err := c.queryBox.FindString("broadcast_session_info.tmpl.sql")
	if err != nil {
		return nil, err
	}

	tmpl, err := template.New("play_session_info").Parse(query)
	if err != nil {
		return nil, err
	}

	var output bytes.Buffer

	if !validIngestSessionID.MatchString(params.IngestSessionId) {
		return nil, errors.New("Invalid Ingest Session Id")
	}

	// WARNING, values added here need to be sanitised to prevent SQL injection, BE CAREFUL!!
	vars := broadcastSessionInfoTemplateVars{
		IngestSessionID: params.IngestSessionId,
		StartDateTime:   startDateTime.Format(sqlDateTimeFormat),
		EndDateTime:     endDateTime.Format(sqlDateTimeFormat),
		StartDate:       startDate.Format(sqlDateFormat),
		EndDate:         endDate.Format(sqlDateFormat),
	}

	err = tmpl.Execute(&output, vars)
	if err != nil {
		return nil, err
	}

	// We're using templates rather than prepared statements because we need multiple result sets
	return db.QueryContext(ctx, output.String())
}

func (c *client) mapBroadcastInfoResponse(queryID string, rows *sql.Rows) (*control.GetQueryResultResponse, error) {
	broadcastSesisonInfo := &control.BroadcastInfoResponse{}

	result := &control.GetQueryResultResponse{
		Result: &control.GetQueryResultResponse_BroadcastInfo{
			BroadcastInfo: broadcastSesisonInfo,
		},
	}

	// Iterate over the resultsets and return the data

	// First resultset is Play Sessions by MW
	broadcastSesisonInfo.PlaySessionsByMinutesWatched = make([]*control.PlaySessionByMinuteWatched, 0)

	for rows.Next() {
		var lastMw time.Time
		var eventTime time.Time
		var eventClientTime time.Time

		event := control.PlaySessionByMinuteWatched{
			PlaySessionSummary: &control.PlaySessionSummary{},
		}

		rows.Scan(
			&event.MinutesWatched,                        // minutes_watched
			&lastMw,                                      // last_mw
			&eventTime,                                   // time_utc
			&eventClientTime,                             // client_time_utc
			&event.PlaySessionSummary.PlaySessionId,      // play_session_id
			&event.PlaySessionSummary.BroadcastId,        // bradcast_id
			&event.PlaySessionSummary.ManifestCluster,    // manifest_cluster
			&event.PlaySessionSummary.Node,               // node
			&event.PlaySessionSummary.Asn,                // asn
			&event.PlaySessionSummary.TimeSinceLoadStart, // time_since_load_start
		)

		var err error

		event.PlaySessionSummary.LastMinuteWatchedTime, err = ptypes.TimestampProto(lastMw)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse LastMinuteWatchedTime")
		}

		event.PlaySessionSummary.Time, err = ptypes.TimestampProto(eventTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse Time")
		}

		event.PlaySessionSummary.ClientTime, err = ptypes.TimestampProto(eventClientTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse ClientTime")
		}

		broadcastSesisonInfo.PlaySessionsByMinutesWatched = append(broadcastSesisonInfo.PlaySessionsByMinutesWatched, &event)
	}

	// Next resultset is Play Sessions by BE/MW
	if !rows.NextResultSet() {
		c.logger.Warnf("Expected Play Sessions by BE/MW resultset. Query ID %v", queryID)

		// We'll just return what we have if there's no other data
		return result, nil
	}

	broadcastSesisonInfo.PlaySessionsByBemw = make([]*control.PlaySessionByBEMW, 0)

	for rows.Next() {
		var lastMw time.Time
		var eventTime time.Time
		var eventClientTime time.Time

		event := control.PlaySessionByBEMW{
			PlaySessionSummary: &control.PlaySessionSummary{},
		}

		rows.Scan(
			&event.MinutesWatched,                        // minutes_watched
			&lastMw,                                      // last_mw
			&event.BufferEmpties,                         // buffer_empties
			&event.Bemw,                                  // be_mw
			&eventTime,                                   // time_utc
			&eventClientTime,                             // client_time_utc
			&event.PlaySessionSummary.PlaySessionId,      // play_session_id
			&event.PlaySessionSummary.BroadcastId,        // bradcast_id
			&event.PlaySessionSummary.ManifestCluster,    // manifest_cluster
			&event.PlaySessionSummary.Node,               // node
			&event.PlaySessionSummary.Asn,                // asn
			&event.PlaySessionSummary.TimeSinceLoadStart, // time_since_load_start
		)

		var err error

		event.PlaySessionSummary.LastMinuteWatchedTime, err = ptypes.TimestampProto(lastMw)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse LastMinuteWatchedTime")
		}

		event.PlaySessionSummary.Time, err = ptypes.TimestampProto(eventTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse Time")
		}

		event.PlaySessionSummary.ClientTime, err = ptypes.TimestampProto(eventClientTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse ClientTime")
		}

		broadcastSesisonInfo.PlaySessionsByBemw = append(broadcastSesisonInfo.PlaySessionsByBemw, &event)
	}

	return result, nil
}
