package svc

import (
	"bytes"
	"context"
	"database/sql"
	"errors"
	"regexp"
	"text/template"
	"time"

	control "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	"github.com/golang/protobuf/ptypes"
)

type playSessionInfoTemplateVars struct {
	PlaySessionID string
	StartDateTime string
	EndDateTime   string
	StartDate     string
	EndDate       string
}

var validPlaySessionID = regexp.MustCompile("^[0-9a-fA-F]+$")

func (c *client) formatPlaySessionInfoQuery(ctx context.Context, db *sql.DB, startDateTime, endDateTime, startDate, endDate time.Time, params *control.PlaySessionInfoParameters) (*sql.Rows, error) {
	// Grab the SQL from packr and add the params in
	query, err := c.queryBox.FindString("play_session_info.tmpl.sql")
	if err != nil {
		return nil, err
	}

	tmpl, err := template.New("play_session_info").Parse(query)
	if err != nil {
		return nil, err
	}

	var output bytes.Buffer

	if !validPlaySessionID.MatchString(params.PlaySessionId) {
		return nil, errors.New("Invalid Play Session Id")
	}

	// WARNING, values added here need to be sanitised to prevent SQL injection, BE CAREFUL!!
	vars := playSessionInfoTemplateVars{
		PlaySessionID: params.PlaySessionId,
		StartDateTime: startDateTime.Format(sqlDateTimeFormat),
		EndDateTime:   endDateTime.Format(sqlDateTimeFormat),
		StartDate:     startDate.Format(sqlDateFormat),
		EndDate:       endDate.Format(sqlDateFormat),
	}

	err = tmpl.Execute(&output, vars)
	if err != nil {
		return nil, err
	}

	// We're using templates rather than prepared statements because we need multiple result sets
	return db.QueryContext(ctx, output.String())
}

func (c *client) mapPlaySessionInfoResponse(queryID string, rows *sql.Rows) (*control.GetQueryResultResponse, error) {
	playSessionInfo := &control.PlaySessionInfoResponse{}

	result := &control.GetQueryResultResponse{
		Result: &control.GetQueryResultResponse_PlaySessionInfo{
			PlaySessionInfo: playSessionInfo,
		},
	}

	// Iterate over the resultsets and return the data

	// First resultset is PlaybackSummary
	if rows.Next() {
		playSessionInfo.PlaybackSummary = &control.PlaybackSummary{}

		var firstMwTime time.Time
		var lastMwTime time.Time

		rows.Scan(
			&firstMwTime, // first_mw_time
			&lastMwTime,  // last_mw_time
			&playSessionInfo.PlaybackSummary.MinutesWatched,       // minutes_watched
			&playSessionInfo.PlaybackSummary.BufferEmpties,        // buffer_empties
			&playSessionInfo.PlaybackSummary.Errors,               // errors
			&playSessionInfo.PlaybackSummary.SecondsBuffering,     // seconds_buffering
			&playSessionInfo.PlaybackSummary.PercentTimeBuffering, // percent_time_buffering
			&playSessionInfo.PlaybackSummary.AvgLatency,           // avg_latency
			&playSessionInfo.PlaybackSummary.P50Latency,           // p50_latency
		)

		var err error

		playSessionInfo.PlaybackSummary.FirstMwTime, err = ptypes.TimestampProto(firstMwTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse FirstMwTime")
		}

		playSessionInfo.PlaybackSummary.LastMwTime, err = ptypes.TimestampProto(lastMwTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse LastMwTime")
		}

		// We have to manually call rows.Next() here because we only expect 1 row above
		// and apparently we have to do that to trigger the next resultset
		rows.Next()
	}

	// Next resultset is VideoErrors
	if !rows.NextResultSet() {
		c.logger.Warnf("Expected VideoErrors resultset. Query ID %v", queryID)
		return result, nil
	}

	playSessionInfo.VideoErrors = make([]*control.VideoError, 0)

	for rows.Next() {
		ve := &control.VideoError{}

		var errorTime time.Time

		rows.Scan(
			&errorTime,
			&ve.VideoErrorCode,
			&ve.VideoErrorMessage,
			&ve.VideoErrorRecoverable,
			&ve.VideoErrorResult,
			&ve.VideoErrorSource,
			&ve.VideoErrorValue,
			&ve.VideoBufferSize,
			&ve.BufferEmptyCount,
			&ve.PlayingAd,
			&ve.Url,
			&ve.ErrorExtra,
			&ve.ErrorType,
			&ve.ErrorWhat,
		)

		var err error

		ve.Time, err = ptypes.TimestampProto(errorTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse errorTime")
		}

		playSessionInfo.VideoErrors = append(playSessionInfo.VideoErrors, ve)
	}

	// Next resultset is PlaySessionInfo
	if !rows.NextResultSet() {
		c.logger.Warnf("Expected PlaySessionInfo resultset. Query ID %v", queryID)
		return result, nil
	}

	if rows.Next() {
		playSessionInfo.PlaySessionInfo = &control.PlaySessionInfo{}

		var videoPlayTime time.Time

		rows.Scan(
			&videoPlayTime,
			&playSessionInfo.PlaySessionInfo.Platform,
			&playSessionInfo.PlaySessionInfo.Player,
			&playSessionInfo.PlaySessionInfo.CoreVersion,
			&playSessionInfo.PlaySessionInfo.Backend,
			&playSessionInfo.PlaySessionInfo.BackendVersion,
			&playSessionInfo.PlaySessionInfo.Channel,
			&playSessionInfo.PlaySessionInfo.ContentId,
			&playSessionInfo.PlaySessionInfo.CustomerId,
			&playSessionInfo.PlaySessionInfo.Cluster,
			&playSessionInfo.PlaySessionInfo.Node,
			&playSessionInfo.PlaySessionInfo.ServingId,
			&playSessionInfo.PlaySessionInfo.Country,
			&playSessionInfo.PlaySessionInfo.Live,
			&playSessionInfo.PlaySessionInfo.VodCdnOrigin,
			&playSessionInfo.PlaySessionInfo.LowLatency,
			&playSessionInfo.PlaySessionInfo.ClientApp,
			&playSessionInfo.PlaySessionInfo.AppVersion,
			&playSessionInfo.PlaySessionInfo.BrowserFamily,
			&playSessionInfo.PlaySessionInfo.BrowserVersion,
			&playSessionInfo.PlaySessionInfo.Os,
			&playSessionInfo.PlaySessionInfo.OsName,
			&playSessionInfo.PlaySessionInfo.OsVersion,
			&playSessionInfo.PlaySessionInfo.DeviceManufacturer,
			&playSessionInfo.PlaySessionInfo.TimeForPlayerCoreLoad,
			&playSessionInfo.PlaySessionInfo.TimeForTokenLoad,
			&playSessionInfo.PlaySessionInfo.TimeToMasterPlaylistReady,
			&playSessionInfo.PlaySessionInfo.TimeToVariantReady,
			&playSessionInfo.PlaySessionInfo.TimeToSegmentReady,
			&playSessionInfo.PlaySessionInfo.TimeSinceLoadStart,
			&playSessionInfo.PlaySessionInfo.DeviceModel,
			&playSessionInfo.PlaySessionInfo.DeviceOsVersion,
		)

		var err error

		playSessionInfo.PlaySessionInfo.Time, err = ptypes.TimestampProto(videoPlayTime)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to parse videoPlayTime")
		}

		// We have to manually call rows.Next() here because we only expect 1 row above
		// and apparently we have to do that to trigger the next resultset
		rows.Next()
	}

	return result, nil
}
