package svc

import (
	"context"
	"database/sql"
	"fmt"
	"strings"
	"time"

	"github.com/golang/protobuf/jsonpb"

	// Import postgres driver
	_ "github.com/lib/pq"

	control "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/gobuffalo/packr/v2"
	"github.com/golang/protobuf/ptypes"
	"github.com/sirupsen/logrus"
)

const connStr = "postgres://%s:%s@%s/%s?sslmode=verify-ca"
const sqlDateTimeFormat = "2006-01-02 15:04:05"
const sqlDateFormat = "2006-01-02 00:00:00" // Used on the tahoe date column

// Client defines the functions that will be available in this service, in this case it's pretty much a straight implementation of the twirp service
type Client interface {
	ExecuteQuery(context context.Context, queryID, resultPath string, query *control.Query) error
}

type client struct {
	redshiftAddr      string
	redshiftUser      string
	redshiftPass      string
	redshiftDatabase  string
	resultsBucketName string
	queriesTableName  string
	queryBox          *packr.Box
	s3                *s3.S3
	ddb               *dynamodb.DynamoDB
	logger            logrus.FieldLogger
}

// New returns a new Carrot Analytics client
func New(sess *session.Session, redshiftAddr, redshiftUser, redshiftPass, redshiftDatabase, resultsBucketName, queriesTableName string, queryBox *packr.Box, logger logrus.FieldLogger) Client {
	return &client{
		redshiftAddr:      redshiftAddr,
		redshiftUser:      redshiftUser,
		redshiftPass:      redshiftPass,
		redshiftDatabase:  redshiftDatabase,
		resultsBucketName: resultsBucketName,
		queriesTableName:  queriesTableName,
		queryBox:          queryBox,
		s3:                s3.New(sess),
		ddb:               dynamodb.New(sess),
		logger:            logger,
	}
}

func (c *client) ExecuteQuery(ctx context.Context, queryID, resultPath string, query *control.Query) error {
	db, err := sql.Open("postgres", fmt.Sprintf(connStr, c.redshiftUser, c.redshiftPass, c.redshiftAddr, c.redshiftDatabase))
	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	defer db.Close()

	var queryResult *sql.Rows

	startDate, err := ptypes.Timestamp(query.StartDate)
	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	endDate, err := ptypes.Timestamp(query.EndDate)
	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	// Update the db to EXECUTING
	c.updateQueryStatus(queryID, control.QueryStatus_EXECUTING)

	// Since we chunk up data in tahoe into day slices and those day slices happen in pacific time we need to expand our start / end times to take
	// care of any issues that might arise with time zones or DST weirdness
	startDate4Head := startDate.Add(-24 * time.Hour)
	endDate4Head := endDate.Add(48 * time.Hour) // 48 because of the way that the date column works in tahoe

	switch params := query.QueryParams.(type) {
	case *control.Query_BroadcastInfoParams:
		queryResult, err = c.formatBroadcastInfoQuery(ctx, db, startDate, endDate, startDate4Head, endDate4Head, params.BroadcastInfoParams)
		break
	case *control.Query_PlaySessionInfoParams:
		queryResult, err = c.formatPlaySessionInfoQuery(ctx, db, startDate, endDate, startDate4Head, endDate4Head, params.PlaySessionInfoParams)
		break
	}

	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	var result *control.GetQueryResultResponse

	switch query.QueryParams.(type) {
	case *control.Query_BroadcastInfoParams:
		result, err = c.mapBroadcastInfoResponse(queryID, queryResult)
		break
	case *control.Query_PlaySessionInfoParams:
		result, err = c.mapPlaySessionInfoResponse(queryID, queryResult)
		break
	default:
		return fmt.Errorf("Could not determine query type %v", query.QueryParams)
	}

	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	// Write result to S3
	m := jsonpb.Marshaler{}
	resultString, err := m.MarshalToString(result)
	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	_, err = c.s3.PutObject(&s3.PutObjectInput{
		Key:    &resultPath,
		Bucket: &c.resultsBucketName,
		Body:   strings.NewReader(resultString),
		// Lifecycle rules on the bucket will govern how this gets tidied up
	})

	if err != nil {
		// Update the db to FAILURE
		c.updateQueryStatus(queryID, control.QueryStatus_FAILURE)
		return err
	}

	// Update the db to SUCCESS
	c.updateQueryStatus(queryID, control.QueryStatus_SUCCESS)

	return nil
}

func (c *client) updateQueryStatus(queryID string, status control.QueryStatus) {
	statusVal, err := dynamodbattribute.Marshal(status)
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal query status update for query ID %v", queryID)
	}

	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: &c.queriesTableName,
		Key: map[string]*dynamodb.AttributeValue{
			"query_id": &dynamodb.AttributeValue{S: &queryID},
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":status_val": statusVal,
		},
		ExpressionAttributeNames: map[string]*string{
			"#s": aws.String("status"),
		},
		UpdateExpression:    aws.String("SET #s = :status_val"),
		ConditionExpression: aws.String("attribute_exists(query_id)"),
	})

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to update query status for query ID %v", queryID)
	}
}
