import lambda = require('@aws-cdk/aws-lambda');
import ec2 = require('@aws-cdk/aws-ec2');
import ddb = require('@aws-cdk/aws-dynamodb');
import iam = require('@aws-cdk/aws-iam');
import cdk = require('@aws-cdk/core');

export interface BindleLockConfig {
  systemAccessBindleLockID: string
  toolsAccessBindleLockID: string
  twitchDataBindleLockID: string
}

export interface CCProps {
  iocpHost: string
  iocpStage: string
  iocpArn: string
  carrotStreamAnalysisLambdaARN: string
  carrotAnalyticsControlLambdaARN: string
  carrotSystemHealthLambdaARN: string
  carrotRtmpRecorderLambdaARN: string
  carrotOmnibarLambdaARN: string
  invokeFunctionRoles: string[]
  bindleLockConfig: BindleLockConfig
  isengardDevRole: string
}

export class CarrotControl extends cdk.Construct {
  CCLambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CCProps) {
    super(scope, id);

    // Table for request audits
    const auditTable = new ddb.Table(this, "CarrotControlAuditTable", {
      tableName: 'carrot-control-audits',
      partitionKey: {
        type: ddb.AttributeType.STRING,
        name: 'audit_id',
      },
      sortKey: {
        name: 'time',
        type: ddb.AttributeType.STRING,
      },
      timeToLiveAttribute: 'ttl',
      billingMode: ddb.BillingMode.PAY_PER_REQUEST,
    })

    auditTable.addGlobalSecondaryIndex({
      indexName: 'carrot-control-audits-client-id',
      partitionKey: {
        name: 'client_id',
        type: ddb.AttributeType.STRING,
      },
      sortKey: {
        name: 'time',
        type: ddb.AttributeType.STRING,
      },
      projectionType: ddb.ProjectionType.ALL,
    })

    auditTable.addGlobalSecondaryIndex({
      indexName: 'carrot-control-audits-user-identifier',
      partitionKey: {
        name: 'user_identifier',
        type: ddb.AttributeType.STRING,
      },
      sortKey: {
        name: 'time',
        type: ddb.AttributeType.STRING,
      },
      projectionType: ddb.ProjectionType.ALL,
    })

    this.CCLambda = new lambda.Function(this, 'CarrotControl', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(20),
      handler: 'bin/linux/carrot-control',
      code: lambda.Code.asset('../bin/carrot-control.zip'),
      environment: {
        "auditTableName": auditTable.tableName,
        "csaLambdaARN": props.carrotStreamAnalysisLambdaARN,
        "cacLambdaARN": props.carrotAnalyticsControlLambdaARN,
        "cshLambdaARN": props.carrotSystemHealthLambdaARN,
        "crrLambdaARN": props.carrotRtmpRecorderLambdaARN,
        "omnibarLambdaARN": props.carrotOmnibarLambdaARN,
        "iocpHost": props.iocpHost,
        "iocpStage": props.iocpStage,
        "systemAccessBindleLockID": props.bindleLockConfig.systemAccessBindleLockID,
        "toolsAccessBindleLockID": props.bindleLockConfig.toolsAccessBindleLockID,
        "twitchDataBindleLockID": props.bindleLockConfig.twitchDataBindleLockID,
      }
    })

    // Allow the service to call BRASS
    this.CCLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "brassservice:IsAuthorized",
        "brassservice:BatchIsAuthorized",
      ],
      resources: [
        '*',
      ]
    }))

    // Grand the Lambda Role access to dynamodb
    this.CCLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "dynamodb:PutItem",
      ],
      resources: [
        auditTable.tableArn,
      ]
    }));

    for (let roleArn of props.invokeFunctionRoles) {
      this.CCLambda.grantInvoke(new iam.ArnPrincipal(roleArn))
    }

    // Since Carrot Analytics is in another account we need to allow it here
    this.CCLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "lambda:InvokeFunction",
      ],
      resources: [
        props.carrotAnalyticsControlLambdaARN,
      ]
    }))

    // Grant the Lambda Role access to call api-execute on IOCP
    this.CCLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "execute-api:Invoke",
      ],
      resources: [
        props.iocpArn,
      ]
    }))

    // Allow local dev to invoke the function
    if (props.isengardDevRole) {
      const isengardRole = iam.Role.fromRoleArn(this, 'IsengardDevRole', props.isengardDevRole)

      isengardRole.addToPrincipalPolicy( new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ['lambda:InvokeFunction'],
        resources: [this.CCLambda.functionArn],
      }))
    }
  }
}
