package main

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"code.justin.tv/coleiain/isengard-dev-helper/pkg/credentials"
	cacSvc "code.justin.tv/event-engineering/carrot-analytics/control/svc"
	"code.justin.tv/event-engineering/carrot-control/pkg/svc"
	omnibar "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	rtmpSvc "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/svc"
	"code.justin.tv/event-engineering/carrot-stream-analysis/pkg/mmdb"
	csaSvc "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/svc"
	cshSvc "code.justin.tv/event-engineering/carrot-system-health/pkg/svc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/sirupsen/logrus"
)

var (
	awsAccountID = "134277936734" // starfruit-support+carrot-stg@amazon.com
	awsRole      = "Admin"
	awsRegion    = "us-west-2"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	credentials, err := credentials.GetIsengardCredentials(awsAccountID, awsRole)

	if err != nil {
		panic(err)
	}

	sess := session.New(&aws.Config{
		Credentials: credentials,
		Region:      &awsRegion,
	})

	// Streamlog
	streamlogEndpoint := "http://streamlog-query-prod.elasticbeanstalk.com"

	// Download the MMDB files using make mmdb
	// This command should be run from the root of the repository
	cityDbPath := "mmdb/GeoIP2-City.mmdb"
	ispDbPath := "mmdb/GeoIP2-ISP.mmdb"

	mmdbClient := mmdb.New(ispDbPath, cityDbPath, logger)
	flvAnalyserBucketName := "carrot-flv-analyser-recordings-stg"
	s3Client := s3.New(sess)

	// Create the dependencies for the service, in lambda this would be a lambda transport twirp client, but for dev testing we're just going
	// to create the service directly from the Carrot Stream Analysis package and use that
	csaServer := csaSvc.New(streamlogEndpoint, mmdbClient, s3Client, flvAnalyserBucketName, logger)

	// Carrot Analytics
	executorSQSQueueURL := "https://sqs.us-west-2.amazonaws.com/291428775132/carrot-analytics-prod-CarrotAnalyticsCarrotAnalyticsExecutorQueueC-Z1STG1KUQZH9"
	queriesTableName := "carrot-analytics-prod-CarrotAnalyticsCarrotAnalyticsQueries56AF22A9-12URBRWLGC5C0"
	resultsBucketName := "carrot-analytics-prod-carrotanalyticscarrotanalyt-1gt1xiojqbdnm"

	// Create the Service
	analyticsControlServer := cacSvc.New(sess, executorSQSQueueURL, queriesTableName, resultsBucketName, logger)

	omenHost := "http://omen.us-west-2.dev.gns.justin.tv/api/plyql/v1/query"
	druidBroker := "druid-broker-0-9-0-rc3.us-west-2.prod.gns.justin.tv:8082"
	omenAppName := "Carrot-control-localdev"
	broodAPIURL := "http://alb.us-west-2.prod.brood.live-video.a2z.com"
	liveProxyAPIURL := "http://liveproxyapi-prod.elasticbeanstalk.com"
	originDCs := "sjc02,cmh01,lhr05,pdx05"
	cshServer, err := cshSvc.New(omenHost, druidBroker, omenAppName, broodAPIURL, liveProxyAPIURL, strings.Split(originDCs, ","), logger)

	if err != nil {
		panic(err)
	}

	// Omnibar
	// Just use the lambda function because I don't want this to need access to the weaver keys
	lambdaClient := lambda.New(sess)
	omnibarLambdaARN := "arn:aws:lambda:us-west-2:134277936734:function:carrot-omnibar-stg-OmnibarFunction2F151B77-M9L191IQKN42"
	lambdaTransport := twirpLambda.NewClient(lambdaClient, omnibarLambdaARN)

	omnibarClient := omnibar.NewCarrotOmnibarProtobufClient("https://would.you.like.a.carrot.twitch.tv", lambdaTransport)

	// IOCP
	iocpConfig := svc.IOCPConfig{
		Host:        "iocp.live-video.a2z.com",
		Stage:       "prod",
		Credentials: sess.Config.Credentials, // Since IOCP allows invocation from anything in our account, our isengard credentials are sufficient here
	}

	// Carrot RTMP Recorder
	availableRegions := []string{"us-west-2"}
	digestionEndpoints := []string{
		"http://digestion.video.justin.tv",
		"http://digestion-cmh01.video.justin.tv",
		"http://direct.eu-west-2.prod.digestion.live-video.a2z.com",
	}
	endpointsTableName := "carrot-rtmp-recorder-api-stg-APIEndopintsTableD1E808A6-1RNAIDSYKX9OP"
	rtmpDumpsTableName := "carrot-rtmp-dumps"
	rtmpDumpQueueURL := "https://sqs.us-west-2.amazonaws.com/134277936734/rtmp-dump-queue"
	resourceGroupName := "carrot-rtmp-recorder-resource-group"
	resourceGroupValue := "would you like a carrot"
	ddbClient := dynamodb.New(sess)
	sqsClient := sqs.New(sess)

	rtmpRecorder := rtmpSvc.New(
		availableRegions,
		digestionEndpoints,
		endpointsTableName,
		rtmpDumpsTableName,
		rtmpDumpQueueURL,
		resourceGroupName,
		resourceGroupValue,
		ddbClient,
		s3Client,
		sqsClient,
		csaServer,
		logger,
	)

	// Create the Carrot Control Service
	ccServer := svc.New(csaServer, analyticsControlServer, cshServer, rtmpRecorder, omnibarClient, iocpConfig, nil, nil, svc.BindleLockConfig{}, logger)

	// Test your shit
	result, err := ccServer.GetPoPHealth(context.Background(), nil)

	if err != nil {
		panic(err)
	}

	bytes, _ := json.MarshalIndent(result, "", "  ")
	fmt.Println(string(bytes))
}
