package main

import (
	"fmt"
	"net/http"
	"os"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	cac "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	aud "code.justin.tv/event-engineering/carrot-control/pkg/auditor"
	cc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-control/pkg/svc"
	omnibar "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	crr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	csh "code.justin.tv/event-engineering/carrot-system-health/pkg/rpc"
	"code.justin.tv/video/brassclient"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/sirupsen/logrus"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	// Create an AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(os.Getenv("AWS_REGION")),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	lambdaClient := lambda.New(sess)

	// Create the Carrot Stream Analysis client
	csaLambdaARN := os.Getenv("csaLambdaARN")
	lambdaTransport := twirpLambda.NewClient(lambdaClient, csaLambdaARN)
	csaClient := csa.NewCarrotStreamAnalysisProtobufClient("https://would.you.like.a.carrot.twitch.tv", lambdaTransport)

	// Create the Carrot Analytics client
	cacLambdaARN := os.Getenv("cacLambdaARN")
	cacLambdaTransport := twirpLambda.NewClient(lambdaClient, cacLambdaARN)
	cacClient := cac.NewCarrotAnalyticsControlProtobufClient("https://would.you.like.a.carrot.twitch.tv", cacLambdaTransport)

	// Create the Carrot System Health client
	cshLambdaARN := os.Getenv("cshLambdaARN")
	cshLambdaTransport := twirpLambda.NewClient(lambdaClient, cshLambdaARN)
	cshClient := csh.NewCarrotSystemHealthProtobufClient("https://would.you.like.a.carrot.twitch.tv", cshLambdaTransport)

	// Create the Carrot RTMP Recorder client
	crrLambdaARN := os.Getenv("crrLambdaARN")
	crrLambdaTransport := twirpLambda.NewClient(lambdaClient, crrLambdaARN)
	crrClient := crr.NewCarrotRtmpRecorderProtobufClient("https://would.you.like.a.carrot.twitch.tv", crrLambdaTransport)

	// Create the Carrot Omnibar client
	omnibarLambdaARN := os.Getenv("omnibarLambdaARN")
	omnibarLambdaTransport := twirpLambda.NewClient(lambdaClient, omnibarLambdaARN)
	omnibarClient := omnibar.NewCarrotOmnibarProtobufClient("https://would.you.like.a.carrot.twitch.tv", omnibarLambdaTransport)

	// Create the IOCP Config
	iocpConfig := svc.IOCPConfig{
		Host:        os.Getenv("iocpHost"),
		Stage:       os.Getenv("iocpStage"),
		Credentials: sess.Config.Credentials,
	}

	auditTableName := os.Getenv("auditTableName")
	auditor := aud.NewDynamoAuditor(sess, auditTableName, logger)

	// We need these bindle lock IDs for managing permissions for certain things
	bindleLockConfig := svc.BindleLockConfig{
		SystemAccessBindleLockID: os.Getenv("systemAccessBindleLockID"),
		ToolsAccessBindleLockID:  os.Getenv("toolsAccessBindleLockID"),
		TwitchDataBindleLockID:   os.Getenv("twitchDataBindleLockID"),
	}
	brassClient := brassclient.New(sess)

	// Create the Service
	ccServer := svc.New(csaClient, cacClient, cshClient, crrClient, omnibarClient, iocpConfig, brassClient, auditor, bindleLockConfig, logger)

	// Create a Twirp service based off the internal service
	err = twirpLambda.ListenAndServe(":80", &headerMapper{
		baseHandler: cc.NewCarrotControlServer(ccServer, nil),
	})

	if err != nil {
		// TODO logging and metrics
		panic(err)
	}
}

// This will map certain http headers to context values so we can get to them inside our service
type headerMapper struct {
	baseHandler http.Handler
}

func (hm *headerMapper) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	ctx := req.Context()

	clientID := req.Header.Get(svc.ClientIDHeader)
	ctx = svc.WithClientID(ctx, clientID)

	auditUserIdentifier := req.Header.Get(svc.AuditUserIDHeader)
	ctx = svc.WithUserID(ctx, auditUserIdentifier)

	req = req.WithContext(ctx)

	hm.baseHandler.ServeHTTP(w, req)
}
