package auditor

import (
	"encoding/json"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/google/uuid"
	"github.com/sirupsen/logrus"
)

// Retain audits for 1 year
const auditTTL = time.Hour * 24 * 7 * 52

// Audit is the data structure for the audit that people provide to us
type Audit struct {
	Endpoint string
	Data     map[string]string
}

// dbAudit is the data structure for the audit that goes into the DB
type dbAudit struct {
	Time     string `json:"time"`
	ClientID string `json:"client_id"`
	UserID   string `json:"user_identifier"`
	Endpoint string `json:"endpoint"`
	Data     string `json:"data"`
	TTL      int64  `json:"ttl"`
}

// Auditor defines the functions that you can perform with the auditor
type Auditor interface {
	CreateAudit(clientID, userID string, audit Audit)
}

type auditor struct {
	dynamo         *dynamodb.DynamoDB
	auditTableName string
	logger         logrus.FieldLogger
}

// NewDynamoAuditor creates an auditor with a dynamodb backend data store
func NewDynamoAuditor(sess *session.Session, auditTableName string, logger logrus.FieldLogger) Auditor {
	return &auditor{
		dynamo:         dynamodb.New(sess),
		auditTableName: auditTableName,
		logger:         logger.WithField("Module", "Auditor"),
	}
}

func (a *auditor) CreateAudit(clientID, userID string, audit Audit) {
	auditID := uuid.New().String()

	// Format the data map to JSON
	data, err := json.Marshal(audit.Data)
	if err != nil {
		a.logger.WithError(err).WithField("audit", audit).Warn("Failed to marshal audit data")
		return
	}

	dbAudit := &dbAudit{
		Time:     time.Now().Format(time.RFC3339),
		ClientID: clientID,
		UserID:   userID,
		Endpoint: audit.Endpoint,
		Data:     string(data),
		TTL:      time.Now().Add(auditTTL).Unix(),
	}

	item, err := dynamodbattribute.MarshalMap(dbAudit)
	if err != nil {
		a.logger.WithError(err).WithField("audit", dbAudit).Warn("Failed to marshal audit")
		return
	}

	item["audit_id"] = &dynamodb.AttributeValue{
		S: &auditID,
	}

	_, err = a.dynamo.PutItem(&dynamodb.PutItemInput{
		TableName: aws.String(a.auditTableName),
		Item:      item,
	})

	if err != nil {
		a.logger.WithError(err).WithField("audit", dbAudit).Warn("Failed to create audit")
		return
	}
}
