package svc

import (
	"context"
	"strings"

	aud "code.justin.tv/event-engineering/carrot-control/pkg/auditor"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
)

// GetChannelSessions is just a direct passthrough to the twirp client for the Carrot Stream Analysis service
func (c *client) GetChannelSessions(ctx context.Context, request *csa.GetChannelSessionsRequest) (*csa.GetChannelSessionsResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetChannelSessions",
		Data: map[string]string{
			"customer_id": request.CustomerId,
			"content_id":  request.ContentId,
		},
	})

	if err != nil {
		return nil, err
	}

	if strings.EqualFold(request.CustomerId, "twitch") {
		// Only certain people can view Twitch data
		err := c.canAccessTwitchData(ctx)

		if err != nil {
			return nil, err
		}
	}

	return c.streamAnalysis.GetChannelSessions(ctx, request)
}

// GetSessionData is just a direct passthrough to the twirp client for the Carrot Stream Analysis service
func (c *client) GetSessionData(ctx context.Context, request *csa.GetSessionDataRequest) (*csa.GetSessionDataResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetSessionData",
		Data: map[string]string{
			"customer_id": request.CustomerId,
			"content_id":  request.ContentId,
			"session_id":  request.SessionId,
		},
	})

	if err != nil {
		return nil, err
	}

	if strings.EqualFold(request.CustomerId, "twitch") {
		// Only certain people can view Twitch data
		err := c.canAccessTwitchData(ctx)

		if err != nil {
			return nil, err
		}
	}

	return c.streamAnalysis.GetSessionData(ctx, request)
}

// GetCapturedFileDetail is just a direct passthrough to the twirp client for the Carrot Stream Analysis service
func (c *client) GetCapturedFileDetail(ctx context.Context, request *csa.GetCapturedFileDetailRequest) (*csa.GetCapturedFileDetailResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetCapturedFileDetail",
		Data: map[string]string{
			"key": request.Key,
		},
	})

	if err != nil {
		return nil, err
	}

	resp, err := c.streamAnalysis.GetCapturedFileDetail(ctx, request)

	if err != nil {
		return nil, err
	}

	// Only certain people can download captured files
	err = c.canAccessTools(ctx)

	if err == nil {
		resp.Detail.CanDownload = true
	}

	return resp, nil
}

// GetCapturedFileDownloadLink is just a direct passthrough to the twirp client for the Carrot Stream Analysis service
func (c *client) GetCapturedFileDownloadLink(ctx context.Context, request *csa.GetCapturedFileDownloadLinkRequest) (*csa.GetCapturedFileDownloadLinkResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetCapturedFileDownloadLink",
		Data: map[string]string{
			"key": request.Key,
		},
	})

	if err != nil {
		return nil, err
	}

	// Only certain people can download captured files
	err = c.canAccessTools(ctx)

	if err != nil {
		return nil, err
	}

	return c.streamAnalysis.GetCapturedFileDownloadLink(ctx, request)
}
