package svc

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"sort"
	"time"

	iocp "code.justin.tv/amzn/StarfruitIOCPTwirp"
	sec "code.justin.tv/amzn/StarfruitSECTwirp"
	creds "github.com/aws/aws-sdk-go/aws/credentials"
	v4 "github.com/aws/aws-sdk-go/aws/signer/v4"

	aud "code.justin.tv/event-engineering/carrot-control/pkg/auditor"
	rpc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
)

// IOCPConfig is the configuration object required to call IOCP in multiple regions
type IOCPConfig struct {
	Credentials *creds.Credentials
	Host        string
	Stage       string
}

type httpClient interface {
	Do(req *http.Request) (*http.Response, error)
}

type signingClient struct {
	baseClient httpClient
	signer     *v4.Signer
	service    string
	region     string
}

func (i *signingClient) Do(req *http.Request) (*http.Response, error) {
	b, err := ioutil.ReadAll(req.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to request read body for sigv4 signing: %s", err)
	}

	_, err = i.signer.Sign(req, bytes.NewReader(b), i.service, i.region, time.Now())
	if err != nil {
		return nil, fmt.Errorf("failed to sigv4 sign the request: %s", err)
	}
	return i.baseClient.Do(req)
}

func (c *client) getIOCPClient(region string) (endpoint string, client *signingClient) {
	endpoint = fmt.Sprintf("https://%s.%s.%s", region, c.iocpConfig.Stage, c.iocpConfig.Host)

	client = &signingClient{
		baseClient: &http.Client{Transport: http.DefaultTransport},
		signer:     v4.NewSigner(c.iocpConfig.Credentials),
		service:    "execute-api",
		region:     region,
	}

	return
}

// GetStreamConfig calls the IOCP endpoint for the region specified and returns the result
func (c *client) GetStreamConfig(ctx context.Context, request *rpc.GetStreamConfigRequest) (*rpc.GetStreamConfigResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetStreamConfig",
		Data: map[string]string{
			"customer_id":      request.Request.CustomerId,
			"region":           request.Region,
			"stream_config_id": request.Request.StreamConfigId,
		},
	})

	if err != nil {
		return nil, err
	}

	streamConfig, err := iocp.NewStreamConfigAPIProtobufClient(c.getIOCPClient(request.Region)).GetStreamConfig(ctx, request.Request)

	if err != nil {
		c.logger.WithField("endpoint", "GetStreamConfig").WithError(err).Warnf("Failed to get stream config with id %v", request.Request.StreamConfigId)
		return nil, err
	}

	streamKeys, err := iocp.NewStreamKeyAPIProtobufClient(c.getIOCPClient(request.Region)).ListStreamKeysForContent(ctx, &iocp.ListStreamKeysForContentRequest{
		CustomerId: request.Request.CustomerId,
		ContentId:  request.Request.StreamConfigId,
	})

	// We aint in the business of secrets
	for _, key := range streamKeys.StreamKeys {
		key.Secret = ""
	}

	if err != nil {
		c.logger.WithField("endpoint", "GetStreamConfig").WithError(err).Warnf("Failed to get stream keys for stream config %v", request.Request.StreamConfigId)
		return nil, err
	}

	return &rpc.GetStreamConfigResponse{
		StreamConfig: streamConfig,
		StreamKeys:   streamKeys.StreamKeys,
	}, nil
}

// ListStreamConfigs calls the IOCP endpoint for the region specified and returns the result
func (c *client) ListStreamConfigs(ctx context.Context, request *rpc.ListStreamConfigsRequest) (*iocp.ListStreamConfigsResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListStreamConfigs",
		Data: map[string]string{
			"customer_id": request.Request.CustomerId,
			"region":      request.Region,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewStreamConfigAPIProtobufClient(c.getIOCPClient(request.Region)).ListStreamConfigs(ctx, request.Request)
}

// ListStreamConfigsByRecordingConfig calls the IOCP endpoint for the region specified and returns the result
func (c *client) ListStreamConfigsByRecordingConfig(ctx context.Context, request *rpc.ListStreamConfigsByRecordingConfigRequest) (*iocp.ListStreamConfigsByRecordingConfigResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListStreamConfigsByRecordingConfig",
		Data: map[string]string{
			"customer_id":         request.Request.CustomerId,
			"region":              request.Region,
			"recording_config_id": request.Request.RecordingConfigId,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewStreamConfigAPIProtobufClient(c.getIOCPClient(request.Region)).ListStreamConfigsByRecordingConfig(ctx, request.Request)
}

// GetStreamKey calls the IOCP endpoint for the region specified and returns the result
func (c *client) GetStreamKey(ctx context.Context, request *rpc.GetStreamKeyRequest) (*iocp.StreamKey, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetStreamKey",
		Data: map[string]string{
			"region":        request.Region,
			"stream_key_id": request.Request.Id,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewStreamKeyAPIProtobufClient(c.getIOCPClient(request.Region)).GetStreamKey(ctx, request.Request)
}

// ListStreamKeysForContent calls the IOCP endpoint for the region specified and returns the result
func (c *client) ListStreamKeysForContent(ctx context.Context, request *rpc.ListStreamKeysForContentRequest) (*iocp.ListStreamKeysForContentResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListStreamKeysForContent",
		Data: map[string]string{
			"customer_id": request.Request.CustomerId,
			"region":      request.Region,
			"content_id":  request.Request.ContentId,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewStreamKeyAPIProtobufClient(c.getIOCPClient(request.Region)).ListStreamKeysForContent(ctx, request.Request)
}

// GetPlaybackKey calls the IOCP endpoint for the region specified and returns the result
func (c *client) GetPlaybackKey(ctx context.Context, request *rpc.GetPlaybackKeyRequest) (*iocp.GetPlaybackKeyResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetPlaybackKey",
		Data: map[string]string{
			"customer_id":     request.Request.CustomerId,
			"region":          request.Region,
			"playback_key_id": request.Request.KeyId,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewPlaybackKeyAPIProtobufClient(c.getIOCPClient(request.Region)).GetPlaybackKey(ctx, request.Request)
}

// ListPlaybackKeys calls the IOCP endpoint for the region specified and returns the result
func (c *client) ListPlaybackKeys(ctx context.Context, request *rpc.ListPlaybackKeysRequest) (*iocp.ListPlaybackKeysResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListPlaybackKeys",
		Data: map[string]string{
			"customer_id": request.Request.CustomerId,
			"region":      request.Region,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewPlaybackKeyAPIProtobufClient(c.getIOCPClient(request.Region)).ListPlaybackKeys(ctx, request.Request)
}

// GetPlaybackAuthorization calls the IOCP endpoint for the region specified and returns the result
func (c *client) GetPlaybackAuthorization(ctx context.Context, request *rpc.GetPlaybackAuthorizationRequest) (*iocp.GetPlaybackAuthorizationResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetPlaybackAuthorization",
		Data: map[string]string{
			"customer_id":      request.Request.CustomerId,
			"region":           request.Region,
			"stream_config_id": request.Request.StreamConfigId,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewPlaybackKeyAPIProtobufClient(c.getIOCPClient(request.Region)).GetPlaybackAuthorization(ctx, request.Request)
}

// ListStreamEvents calls the IOCP endpoint for the region specified and returns the result
func (c *client) ListStreamEvents(ctx context.Context, request *rpc.ListStreamEventsRequest) (*sec.ListStreamEventsResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListStreamEvents",
		Data: map[string]string{
			"channel_arn":          request.Request.ChannelArn,
			"region":               request.Region,
			"broadcast_session_id": request.Request.BroadcastSessionId,
		},
	})

	if err != nil {
		return nil, err
	}

	resp, err := iocp.NewStreamManagementAPIProtobufClient(c.getIOCPClient(request.Region)).ListStreamEvents(ctx, request.Request)

	if err != nil {
		return nil, err
	}

	sort.Slice(resp.StreamEvents, func(i, j int) bool {
		return resp.StreamEvents[i].EventTime.AsTime().Before(resp.StreamEvents[j].EventTime.AsTime())
	})

	return resp, nil
}

// ListLimitBreachEvents calls the IOCP endpoint for the region specified and returns the result
func (c *client) ListLimitBreachEvents(ctx context.Context, request *rpc.ListLimitBreachEventsRequest) (*sec.ListLimitBreachEventsResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListLimitBreachEvents",
		Data: map[string]string{
			"customer_id": request.Request.CustomerId,
			"region":      request.Region,
		},
	})

	if err != nil {
		return nil, err
	}

	return iocp.NewStreamManagementAPIProtobufClient(c.getIOCPClient(request.Region)).ListLimitBreachEvents(ctx, request.Request)
}
