package svc

import (
	"context"
	"strings"

	aud "code.justin.tv/event-engineering/carrot-control/pkg/auditor"
	crr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
)

// GetAvailableCaptureEndpointRegions is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) GetAvailableCaptureEndpointRegions(ctx context.Context, request *crr.GetAvailableCaptureEndpointRegionsRequest) (*crr.AvailableCaptureEndpointRegions, error) {
	// We're not auditing this request because it seems pointless
	err := c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	return c.rtmpRecorder.GetAvailableCaptureEndpointRegions(ctx, request)
}

// CreateCaptureEndpoint is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) CreateCaptureEndpoint(ctx context.Context, request *crr.CreateCaptureEndpointRequest) (*crr.CaptureEndpointInfo, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "CreateCaptureEndpoint",
		Data: map[string]string{
			"region": request.Region,
			"name":   request.Name,
		},
	})

	if err != nil {
		return nil, err
	}

	err = c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	return c.rtmpRecorder.CreateCaptureEndpoint(ctx, request)
}

// GetCaptureEndpoint is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) GetCaptureEndpoint(ctx context.Context, request *crr.GetCaptureEndpointRequest) (*crr.CaptureEndpointDetail, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetCaptureEndpoint",
		Data: map[string]string{
			"id": request.Id,
		},
	})

	if err != nil {
		return nil, err
	}

	err = c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	return c.rtmpRecorder.GetCaptureEndpoint(ctx, request)
}

// ListCaptureEndpoints is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) ListCaptureEndpoints(ctx context.Context, request *crr.ListCaptureEndpointsRequest) (*crr.ListCaptureEndpointsResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListCaptureEndpoints",
		Data:     map[string]string{
			// I think there will be stuff in here soon
		},
	})

	if err != nil {
		return nil, err
	}

	err = c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	return c.rtmpRecorder.ListCaptureEndpoints(ctx, request)
}

// CreateRTMPDump is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) CreateRTMPDump(ctx context.Context, request *crr.CreateRTMPDumpRequest) (*crr.CreateRTMPDumpResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "CreateRTMPDump",
		Data: map[string]string{
			"region":      request.Region,
			"customer_id": request.CustomerId,
			"channel_id":  request.ChannelId,
			"name":        request.Name,
			"format":      request.Format,
		},
	})

	if err != nil {
		return nil, err
	}

	err = c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	if strings.EqualFold(request.CustomerId, "twitch") {
		// Only certain people can view Twitch data
		err := c.canAccessTwitchData(ctx)

		if err != nil {
			return nil, err
		}
	}

	return c.rtmpRecorder.CreateRTMPDump(ctx, request)
}

// GetRTMPDump is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) GetRTMPDump(ctx context.Context, request *crr.GetRTMPDumpRequest) (*crr.RTMPDumpSummary, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "GetRTMPDump",
		Data: map[string]string{
			"rtmp_dump_id": request.RtmpDumpId,
		},
	})

	if err != nil {
		return nil, err
	}

	err = c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	return c.rtmpRecorder.GetRTMPDump(ctx, request)
}

// ListRTMPDumps is just a direct passthrough to the twirp client for the Carrot RTMP Recorder service
func (c *client) ListRTMPDumps(ctx context.Context, request *crr.ListRTMPDumpsRequest) (*crr.ListRTMPDumpsResponse, error) {
	err := c.WriteAudit(ctx, aud.Audit{
		Endpoint: "ListRTMPDumps",
		Data:     map[string]string{
			// I think there will be stuff in here soon
		},
	})

	if err != nil {
		return nil, err
	}

	err = c.canAccessTools(ctx)
	if err != nil {
		return nil, err
	}

	return c.rtmpRecorder.ListRTMPDumps(ctx, request)
}
