package svc

import (
	"context"
	"fmt"

	iocp "code.justin.tv/amzn/StarfruitIOCPTwirp"
	sec "code.justin.tv/amzn/StarfruitSECTwirp"
	cac "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	omnibar "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	crr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	csh "code.justin.tv/event-engineering/carrot-system-health/pkg/rpc"

	aud "code.justin.tv/event-engineering/carrot-control/pkg/auditor"
	rpc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"

	"code.justin.tv/video/brassclient"
	"github.com/sirupsen/logrus"
	"github.com/twitchtv/twirp"
)

// Client defines the functions that will be available in this service, in this case it's pretty much a straight implementation of the twirp service
type Client interface {
	GetModuleAccess(context context.Context, request *rpc.GetModuleAccessRequest) (*rpc.GetModuleAccessResponse, error)
	// Carrot Stream Analysis
	GetChannelSessions(context context.Context, request *csa.GetChannelSessionsRequest) (*csa.GetChannelSessionsResponse, error)
	GetSessionData(context context.Context, request *csa.GetSessionDataRequest) (*csa.GetSessionDataResponse, error)
	GetCapturedFileDetail(context context.Context, request *csa.GetCapturedFileDetailRequest) (*csa.GetCapturedFileDetailResponse, error)
	GetCapturedFileDownloadLink(ctx context.Context, request *csa.GetCapturedFileDownloadLinkRequest) (*csa.GetCapturedFileDownloadLinkResponse, error)
	// IOCP
	GetStreamConfig(context context.Context, request *rpc.GetStreamConfigRequest) (*rpc.GetStreamConfigResponse, error)
	ListStreamConfigs(context context.Context, request *rpc.ListStreamConfigsRequest) (*iocp.ListStreamConfigsResponse, error)
	ListStreamConfigsByRecordingConfig(context context.Context, request *rpc.ListStreamConfigsByRecordingConfigRequest) (*iocp.ListStreamConfigsByRecordingConfigResponse, error)
	GetStreamKey(context context.Context, request *rpc.GetStreamKeyRequest) (*iocp.StreamKey, error)
	ListStreamKeysForContent(context context.Context, request *rpc.ListStreamKeysForContentRequest) (*iocp.ListStreamKeysForContentResponse, error)
	GetPlaybackKey(context context.Context, request *rpc.GetPlaybackKeyRequest) (*iocp.GetPlaybackKeyResponse, error)
	ListPlaybackKeys(context context.Context, request *rpc.ListPlaybackKeysRequest) (*iocp.ListPlaybackKeysResponse, error)
	GetPlaybackAuthorization(context context.Context, request *rpc.GetPlaybackAuthorizationRequest) (*iocp.GetPlaybackAuthorizationResponse, error)
	ListStreamEvents(ctx context.Context, request *rpc.ListStreamEventsRequest) (*sec.ListStreamEventsResponse, error)
	ListLimitBreachEvents(ctx context.Context, request *rpc.ListLimitBreachEventsRequest) (*sec.ListLimitBreachEventsResponse, error)
	// Carrot Analytics
	EnqueueQuery(context context.Context, request *cac.EnqueueQueryRequest) (*cac.EnqueueQueryResponse, error)
	GetQueryResult(context context.Context, request *cac.GetQueryResultRequest) (*cac.GetQueryResultResponse, error)
	ListQueries(context context.Context, request *cac.ListQueriesRequest) (*cac.ListQueriesResponse, error)
	// Carrot System Health
	GetPoPHealth(context context.Context, request *csh.GetPoPHealthRequest) (*csh.GetPoPHealthResponse, error)
	// Carrot RTMP Recorder
	GetAvailableCaptureEndpointRegions(context context.Context, request *crr.GetAvailableCaptureEndpointRegionsRequest) (*crr.AvailableCaptureEndpointRegions, error)
	CreateCaptureEndpoint(context context.Context, request *crr.CreateCaptureEndpointRequest) (*crr.CaptureEndpointInfo, error)
	GetCaptureEndpoint(context context.Context, request *crr.GetCaptureEndpointRequest) (*crr.CaptureEndpointDetail, error)
	ListCaptureEndpoints(context context.Context, request *crr.ListCaptureEndpointsRequest) (*crr.ListCaptureEndpointsResponse, error)
	CreateRTMPDump(ctx context.Context, request *crr.CreateRTMPDumpRequest) (*crr.CreateRTMPDumpResponse, error)
	GetRTMPDump(ctx context.Context, request *crr.GetRTMPDumpRequest) (*crr.RTMPDumpSummary, error)
	ListRTMPDumps(ctx context.Context, request *crr.ListRTMPDumpsRequest) (*crr.ListRTMPDumpsResponse, error)
	// Omnibar
	WhatIs(ctx context.Context, request *omnibar.This) (*omnibar.Thing, error)
}

type client struct {
	streamAnalysis csa.CarrotStreamAnalysis
	analytics      cac.CarrotAnalyticsControl
	health         csh.CarrotSystemHealth
	rtmpRecorder   crr.CarrotRtmpRecorder
	omnibar        omnibar.CarrotOmnibar
	iocpConfig     IOCPConfig
	brass          *brassclient.BRASS
	auditor        aud.Auditor
	logger         logrus.FieldLogger
	bindleLocks    BindleLockConfig
}

// Keep all the bindle config together to make it a bit easier to manage
type BindleLockConfig struct {
	SystemAccessBindleLockID string
	ToolsAccessBindleLockID  string
	TwitchDataBindleLockID   string
}

// New returns a new Carrot Stream Control client
func New(streamAnalysis csa.CarrotStreamAnalysis, analytics cac.CarrotAnalyticsControl, health csh.CarrotSystemHealth, rtmpRecorder crr.CarrotRtmpRecorder, omnibar omnibar.CarrotOmnibar, iocpConfig IOCPConfig, brass *brassclient.BRASS, auditor aud.Auditor, bindleLocks BindleLockConfig, logger logrus.FieldLogger) Client {
	return &client{
		streamAnalysis: streamAnalysis,
		analytics:      analytics,
		health:         health,
		rtmpRecorder:   rtmpRecorder,
		omnibar:        omnibar,
		iocpConfig:     iocpConfig,
		brass:          brass,
		auditor:        auditor,
		logger:         logger,
		bindleLocks:    bindleLocks,
	}
}

// WriteAudit will only error if we don't have the information required to write an audit, not if the audit fails to be written
func (c *client) WriteAudit(ctx context.Context, audit aud.Audit) error {
	if c.auditor == nil {
		return nil
	}

	// Get the client ID and audit user ID from the context
	clientID := ClientID(ctx)
	userID := UserID(ctx)

	if clientID == "" {
		return twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a client id in the header %v", ClientIDHeader))
	}

	if userID == "" {
		return twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", AuditUserIDHeader))
	}

	go c.auditor.CreateAudit(clientID, userID, audit)

	return nil
}

func (c *client) HasPermission(ctx context.Context, bindleLockID string) (bool, error) {
	// This happens when testing on dev so just allow it
	if c.brass == nil || bindleLockID == "" {
		return true, nil
	}

	userID := UserID(ctx)
	if userID == "" {
		return false, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", AuditUserIDHeader))
	}

	resp, err := c.brass.IsAuthorized(context.Background(), &brassclient.IsAuthorizedRequest{
		Operation: brassclient.OperationUnlock,
		Actor: &brassclient.ActorReference{
			Type: brassclient.ActorTypePrincipal,
			ID:   userID,
		},
		Resource: &brassclient.ResourceReference{
			Type:      brassclient.ResourceTypeLock,
			Namespace: brassclient.NamespaceBindle,
			Name:      bindleLockID,
		},
	})

	if err != nil {
		return false, fmt.Errorf("BRASS IsAuthorized user=%q lock=%q: %w", userID, bindleLockID, err)
	}

	return resp.Authorized, nil
}

func (c *client) GetModuleAccess(ctx context.Context, request *rpc.GetModuleAccessRequest) (*rpc.GetModuleAccessResponse, error) {
	// This happens when testing on dev so just allow it
	if c.brass == nil {
		return &rpc.GetModuleAccessResponse{
			CanAccessSystem:     true,
			CanAccessTools:      true,
			CanAccessTwitchData: true,
		}, nil
	}

	userID := UserID(ctx)
	if userID == "" {
		return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", AuditUserIDHeader))
	}

	// It's important that the order of these matches the order of the bools in the response because... that's how BRASS works
	resources := []*brassclient.ResourceReference{
		&brassclient.ResourceReference{
			Type:      brassclient.ResourceTypeLock,
			Namespace: brassclient.NamespaceBindle,
			Name:      c.bindleLocks.SystemAccessBindleLockID,
		},
		&brassclient.ResourceReference{
			Type:      brassclient.ResourceTypeLock,
			Namespace: brassclient.NamespaceBindle,
			Name:      c.bindleLocks.ToolsAccessBindleLockID,
		},
		&brassclient.ResourceReference{
			Type:      brassclient.ResourceTypeLock,
			Namespace: brassclient.NamespaceBindle,
			Name:      c.bindleLocks.TwitchDataBindleLockID,
		},
	}

	resp, err := c.brass.BatchIsAuthorized(context.Background(), &brassclient.BatchIsAuthorizedRequest{
		Operation: brassclient.OperationUnlock,
		Actor: &brassclient.ActorReference{
			Type: brassclient.ActorTypePrincipal,
			ID:   userID,
		},
		Resources: resources,
	})

	if err != nil {
		return nil, fmt.Errorf("BRASS BatchIsAuthorized user=%q %w", userID, err)
	}

	if len(resp.Authorizations) != len(resources) {
		return nil, fmt.Errorf("Expected %v authorizations, got %v", len(resources), len(resp.Authorizations))
	}

	result := &rpc.GetModuleAccessResponse{}

	for _, authorisationResult := range resp.Authorizations {
		switch authorisationResult.Resource.Name {
		case c.bindleLocks.SystemAccessBindleLockID:
			result.CanAccessSystem = authorisationResult.Authorized
			break
		case c.bindleLocks.ToolsAccessBindleLockID:
			result.CanAccessTools = authorisationResult.Authorized
			break
		case c.bindleLocks.TwitchDataBindleLockID:
			result.CanAccessTwitchData = authorisationResult.Authorized
			break
		}
	}

	return result, nil
}

func (c *client) canAccessTwitchData(ctx context.Context) error {
	hasPermission, err := c.HasPermission(ctx, c.bindleLocks.TwitchDataBindleLockID)

	if err != nil {
		return err
	}

	if !hasPermission {
		return twirp.NewError(twirp.PermissionDenied, "You are not authorised to view Twitch data")
	}

	return nil
}

func (c *client) canAccessTools(ctx context.Context) error {
	hasPermission, err := c.HasPermission(ctx, c.bindleLocks.ToolsAccessBindleLockID)

	if err != nil {
		return err
	}

	if !hasPermission {
		return twirp.NewError(twirp.PermissionDenied, "You are not authorised to perform this action")
	}

	return nil
}
