import lambda = require("@aws-cdk/aws-lambda");
import ec2 = require("@aws-cdk/aws-ec2");
import iam = require("@aws-cdk/aws-iam");
import cdk = require("@aws-cdk/core");

export interface COProps {
    iocpHost: string;
    iocpStage: string;
    iocpArn: string;
    invokeFunctionRoles: string[];
}

export class CarrotOmnibar extends cdk.Construct {
  COLambda: lambda.Function;

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: COProps) {
    super(scope, id);

    this.COLambda = new lambda.Function(this, "Function", {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(5),
      handler: "bin/linux/carrot-omnibar",
      code: lambda.Code.fromAsset("../bin/carrot-omnibar.zip"),
        environment: {
          'iocpHost': props.iocpHost,
          'iocpStage': props.iocpStage,
        }
    });

    // Allow fetching of weaver keys
    this.COLambda.addToRolePolicy(
      new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ["secretsmanager:GetSecretValue"],
        resources: [
          "arn:aws:secretsmanager:*:*:secret:vault/playlist/*",
          "arn:aws:secretsmanager:*:*:secret:vault/segment/*",
        ],
      })
    );

    this.COLambda.addToRolePolicy(
      new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ["kms:Decrypt"],
        resources: ["*"],
        conditions: {
          StringLike: {
            "kms:ViaService": "secretsmanager.*.amazonaws.com",
          },
        },
      })
    );

    for (let roleArn of props.invokeFunctionRoles) {
      this.COLambda.grantInvoke(new iam.ArnPrincipal(roleArn));
    }

    // Grant the Lambda Role access to call api-execute on IOCP
    this.COLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
          "execute-api:Invoke",
      ],
      resources: [
          props.iocpArn,
      ]
    }))
  }
}
