package main

import (
	"context"
	"fmt"
	"os"
	"code.justin.tv/video/weaver-api/auth2"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"
	"code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	rpc "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-omnibar/pkg/svc"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	playlistDecrypter, segmentDecrypter, err := getDecrypters(ctx)
	if err != nil {
		panic(err)
	}

	go func() {
		playlistDecrypter.Run(ctx)
		segmentDecrypter.Run(ctx)
	}()

	// Create an AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(os.Getenv("AWS_REGION")),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	// Create the IOCP Config
	iocpConfig := svc.IOCPConfig{
		Host:        os.Getenv("iocpHost"),
		Stage:       os.Getenv("iocpStage"),
		Credentials: sess.Config.Credentials,
	}

	// Create the Service
	omnibarSvc, err := svc.New(playlistDecrypter, segmentDecrypter, iocpConfig, logger)
	if err != nil {
		panic(err)
	}

	// Create a Twirp service based off the internal service
	server := rpc.NewCarrotOmnibarServer(omnibarSvc, nil)

	err = lambda.ListenAndServe(":80", server)
	if err != nil {
		// TODO logging and metrics
		panic(err)
	}
}

func getDecrypters(ctx context.Context) (playlistDecrypter auth2.Decrypter, segmentDecrypter auth2.Decrypter, err error) {
	aliases := []string{
		auth2.AliasStaging,
		auth2.AliasLegacy,
		auth2.AliasUSWest2,
		auth2.AliasUSEast1,
		auth2.AliasUSEast2,
		auth2.AliasEUWest2,
	}

	playlistDecrypter, err = auth2.NewDecrypter(&auth2.DecrypterConfig{
		RequestType:    auth2.RequestTypePlaylist,
		TrustedAliases: aliases,
	})
	if err != nil {
		return nil, nil, errors.Wrap(err, "unable to create decryptor for playlists")
	}

	segmentDecrypter, err = auth2.NewDecrypter(&auth2.DecrypterConfig{
		RequestType:    auth2.RequestTypeSegment,
		TrustedAliases: aliases,
	})
	if err != nil {
		return nil, nil, errors.Wrap(err, "unable to create decryptor for segments")
	}

	err = playlistDecrypter.Fetch(ctx)
	if err != nil {
		return nil, nil, errors.Wrap(err, "unable to fetch keys for playlists")
	}

	err = segmentDecrypter.Fetch(ctx)
	if err != nil {
		return nil, nil, errors.Wrap(err, "unable to fetch keys for segments")
	}

	return playlistDecrypter, segmentDecrypter, err
}
