package profilehex

import (
	"fmt"
	"regexp"
	"strconv"
)

type Result struct {
	Profile string
	Level   string
}

var h264ProfileLevelHex = regexp.MustCompile(`^avc1.([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})$`)

const (
	ConstraintSetZero  = 128
	ConstraintSetOne   = 6
	ConstraintSetTwo   = 32
	ConstraintSetThree = 16
	ConstraintSetFour  = 8
	ConstraintSetFive  = 4
)

func ParseHexString(input string) (*Result, error) {
	if !h264ProfileLevelHex.MatchString(input) {
		return nil, fmt.Errorf("input was not in the correct format, we expect something that looks like 'avc1.64002A'")
	}

	parts := h264ProfileLevelHex.FindStringSubmatch(input)

	profile, err := strconv.ParseInt(parts[1], 16, 64)

	if err != nil {
		return nil, err
	}

	constraintSet, err := strconv.ParseInt(parts[2], 16, 64)

	if err != nil {
		return nil, err
	}

	level, err := strconv.ParseInt(parts[3], 16, 64)

	if err != nil {
		return nil, err
	}

	result := &Result{
		Profile: "Unknown",
	}

	result.Level = fmt.Sprintf("%.01f", float64(level)/10)

	switch profile {
	case 44:
		result.Profile = "CAVLC 4:4:4 Intra"
		break
	case 66:
		if constraintSet&ConstraintSetOne == ConstraintSetOne {
			result.Profile = "Constrained Baseline"
		} else {
			result.Profile = "Baseline"
		}
		break
	case 77:
		result.Profile = "Main"
		break
	case 83:
		if constraintSet&ConstraintSetFive == ConstraintSetFive {
			result.Profile = "Scalable Constrained Baseline"
		} else {
			result.Profile = "Scalable Baseline"
		}
		break
	case 86:
		if constraintSet&ConstraintSetFive == ConstraintSetFive {
			result.Profile = "Scalable Constrained High"
		} else if constraintSet&ConstraintSetThree == ConstraintSetThree {
			result.Profile = "Scalable High Intra"
		} else {
			result.Profile = "Scalable High"
		}
		break
	case 88:
		result.Profile = "Extended"
		break
	case 100:
		if constraintSet&ConstraintSetFour == ConstraintSetFour {
			result.Profile = "Progressive High"
		} else if constraintSet&(ConstraintSetFour|ConstraintSetFive) == (ConstraintSetFour | ConstraintSetFive) {
			result.Profile = "Constrained High"
		} else {
			result.Profile = "High"
		}
		break
	case 110:
		if constraintSet&ConstraintSetThree == ConstraintSetThree {
			result.Profile = "High 10 Intra"
		} else {
			result.Profile = "High 10"
		}
		break
	case 118:
		result.Profile = "Multiview High"
		break
	case 122:
		if constraintSet&ConstraintSetThree == ConstraintSetThree {
			result.Profile = "High 4:2:2 Intra"
		} else {
			result.Profile = "High 4:2:2"
		}
		break
	case 128:
		result.Profile = "Stereo High"
		break
	case 134:
		result.Profile = "MFC High"
		break
	case 135:
		result.Profile = "MFC Depth High"
		break
	case 138:
		result.Profile = "Multiview Depth High"
		break
	case 139:
		result.Profile = "Enhanced Multiview Depth High"
		break
	case 244:
		if constraintSet&ConstraintSetThree == ConstraintSetThree {
			result.Profile = "High 4:4:4 Intra"
		} else {
			result.Profile = "High 4:4:4 Predictive"
		}
		break
	}

	return result, nil
}
