package svc

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	iocp "code.justin.tv/amzn/StarfruitIOCPTwirp"
	rpc "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	v4 "github.com/aws/aws-sdk-go/aws/signer/v4"
	"github.com/twitchtv/twirp"
)

type httpClient interface {
	Do(req *http.Request) (*http.Response, error)
}

type signingClient struct {
	baseClient httpClient
	signer     *v4.Signer
	service    string
	region     string
}

func (i *signingClient) Do(req *http.Request) (*http.Response, error) {
	b, err := ioutil.ReadAll(req.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to request read body for sigv4 signing: %s", err)
	}

	_, err = i.signer.Sign(req, bytes.NewReader(b), i.service, i.region, time.Now())
	if err != nil {
		return nil, fmt.Errorf("failed to sigv4 sign the request: %s", err)
	}
	return i.baseClient.Do(req)
}

func (c *client) getIOCPClient(region string) (endpoint string, client *signingClient) {
	endpoint = fmt.Sprintf("https://%s.%s.%s", region, c.iocpConfig.Stage, c.iocpConfig.Host)

	client = &signingClient{
		baseClient: &http.Client{Transport: http.DefaultTransport},
		signer:     v4.NewSigner(c.iocpConfig.Credentials),
		service:    "execute-api",
		region:     region,
	}

	return
}

func (c *client) getIVSStreamKey(ctx context.Context, region string, streamKey string) (*rpc.Thing, error) {
	endpoint, iocpClient := c.getIOCPClient(region)

	streamKeyAuthResponse, err := iocp.NewStreamKeyAuthAPIProtobufClient(endpoint, iocpClient).AuthenticateStreamKey(ctx, &iocp.StreamKeyAuthRequest{
		KeyValue: streamKey,
	})

	if err != nil {
		if twErr, ok := err.(twirp.Error); ok && twErr.Code() == twirp.PermissionDenied {
			return &rpc.Thing{
				Type:           rpc.Thing_Generic,
				GenericMessage: "Whilst this looks like a valid stream key, we were unable to retrieve it. This is usually because the stream key has been deleted",
			}, nil
		}

		c.logger.WithError(err).Warnf("Failed to get stream key auth from IOCP")
		return nil, err
	}

	if streamKeyAuthResponse != nil && streamKeyAuthResponse.StreamConfig != nil {
		return &rpc.Thing{
			Type:             rpc.Thing_IVSStreamKey,
			IvsRegion:        region,
			IvsCustomerId:    streamKeyAuthResponse.StreamConfig.CustomerId,
			IvsChannelId:     streamKeyAuthResponse.StreamConfig.StreamConfigId,
			IvsChannelRegion: region,
		}, nil
	}

	return &rpc.Thing{
		Type:           rpc.Thing_Generic,
		GenericMessage: "This is a valid stream key, but no StreamConfig was found, if this persists, please let the IVS Support team know",
	}, nil

}
