package svc

import (
	"context"
	"encoding/json"
	"regexp"
	"strings"

	"code.justin.tv/event-engineering/carrot-omnibar/pkg/profilehex"
	rpc "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	weaverAuth "code.justin.tv/video/weaver-api/auth2"
	creds "github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/sirupsen/logrus"
	"github.com/twitchtv/twirp"
)

// IOCPConfig is the configuration object required to call IOCP in multiple regions
type IOCPConfig struct {
	Credentials *creds.Credentials
	Host        string
	Stage       string
}

// Client defines the functions that will be available in this service, in this case it's pretty much a straight implementation of the twirp service
type Client interface {
	WhatIs(context context.Context, request *rpc.This) (*rpc.Thing, error)
}

type client struct {
	playlistDecrypter weaverAuth.Decrypter
	segmentDecrypter  weaverAuth.Decrypter
	iocpConfig        IOCPConfig
	logger            logrus.FieldLogger
}

// New returns a new Omnibar service
func New(playlistDecrypter weaverAuth.Decrypter, segmentDecrypter weaverAuth.Decrypter, iocpConfig IOCPConfig, logger logrus.FieldLogger) (Client, error) {
	return &client{
		playlistDecrypter: playlistDecrypter,
		segmentDecrypter:  segmentDecrypter,
		iocpConfig:        iocpConfig,
		logger:            logger,
	}, nil
}

var (
	twitchManifestURL     = regexp.MustCompile(`^https?:\/\/usher.ttvnw.net\/api\/channel\/hls\/([^\.]+)\.m3u8\?(.*)$`)
	ivsManifestURL        = regexp.MustCompile(`^https?:\/\/[^\.]+\.([^\.]+)\.playback\.live-video\.net\/api\/video\/v1\/([^.]+)\.([^\.]+)\.channel\.([^\.]+)\.m3u8\??(.*)$`)
	twitchWeaverURLRegex  = regexp.MustCompile(`^https?:\/\/video-weaver\.([^\.]+)\.hls\.ttvnw\.net\/v1\/playlist\/([^\.]+)\.m3u8$`)
	ivsWeaverURLRegex     = regexp.MustCompile(`^https?:\/\/video-weaver\.([^\.]+)\.hls\.live-video\.net\/v1\/playlist\/([^\.]+)\.m3u8$`)
	twitchSegmentURLRegex = regexp.MustCompile(`^https?:\/\/(video-edge-[^.]+)\.([^\.]+)\.(abs|no-abs)\.hls\.ttvnw\.net\/v1\/segment\/([^.]+)\.(ts|fmp4)$`)
	ivsSegmentURLRegex    = regexp.MustCompile(`^https?:\/\/(video-edge-[^.]+)\.([^\.]+)\.hls\.live-video\.net\/v1\/segment\/([^.]+)\.(ts|fmp4)$`)
	twitchStreamKeyRegex  = regexp.MustCompile(`^live_([0-9]+)_[^\?]+\??(.*)$`)
	ivsStreamKeyRegex     = regexp.MustCompile(`^sk_([^_]+)_([^_]+)_[^\?]+\??(.*)$`)
	ivsChannelArn         = regexp.MustCompile(`^arn:aws:ivs:([^:]+):([0-9]+):channel\/(.+)$`)
	h264ProfileLevelHex   = regexp.MustCompile(`^avc1.([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})$`)
)

func (c *client) WhatIs(ctx context.Context, this *rpc.This) (*rpc.Thing, error) {
	this.Thing = strings.TrimSpace(this.Thing)

	if twitchManifestURL.MatchString(this.Thing) {
		matches := twitchManifestURL.FindStringSubmatch(this.Thing)

		return &rpc.Thing{
			Type:          rpc.Thing_TwitchManifestURL,
			TwitchChannel: matches[1],
		}, nil
	}

	if ivsManifestURL.MatchString(this.Thing) {
		matches := ivsManifestURL.FindStringSubmatch(this.Thing)

		return &rpc.Thing{
			Type:             rpc.Thing_IVSManifestURL,
			IvsRegion:        matches[1],
			IvsChannelRegion: matches[2],
			IvsCustomerId:    matches[3],
			IvsChannelId:     matches[4],
		}, nil
	}

	if twitchWeaverURLRegex.MatchString(this.Thing) {
		matches := twitchWeaverURLRegex.FindStringSubmatch(this.Thing)

		_, payload, err := c.playlistDecrypter.Decrypt(matches[2])
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to decode weaver payload from source %v", this.Thing)
			return nil, twirp.NewError(twirp.Internal, "Failed to decode weaver request payload")
		}

		weaverString := ""

		weaverBytes, err := json.Marshal(payload)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to convert weaver payload to JSON")
		} else {
			weaverString = string(weaverBytes)
		}

		return &rpc.Thing{
			Type:              rpc.Thing_TwitchWeaverURL,
			WeaverCluster:     matches[1],
			TwitchChannel:     payload.Session.AnalyticsData.ChannelName,
			TwitchChannelId:   payload.Session.AnalyticsData.ContentId,
			TwitchUserId:      payload.Session.UserId,
			WeaverPayloadJson: weaverString,
		}, nil
	}

	if ivsWeaverURLRegex.MatchString(this.Thing) {
		matches := ivsWeaverURLRegex.FindStringSubmatch(this.Thing)

		_, payload, err := c.playlistDecrypter.Decrypt(matches[2])
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to decode weaver payload from source %v", this.Thing)
			return nil, twirp.NewError(twirp.Internal, "Failed to decode weaver request payload")
		}

		// aws.ivs.us-west-2.298083573632.channel.hdviye1zVPxT
		rawChannel := strings.Split(payload.Session.AssignmentRequest.Stream.Channel, ".")

		weaverString := ""

		weaverBytes, err := json.Marshal(payload)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to convert weaver payload to JSON")
		} else {
			weaverString = string(weaverBytes)
		}

		return &rpc.Thing{
			Type:              rpc.Thing_IVSWeaverURL,
			WeaverCluster:     matches[1],
			IvsRegion:         rawChannel[2],
			IvsCustomerId:     rawChannel[3],
			IvsChannelId:      rawChannel[5],
			WeaverPayloadJson: weaverString,
		}, nil
	}

	if twitchSegmentURLRegex.MatchString(this.Thing) {
		matches := twitchSegmentURLRegex.FindStringSubmatch(this.Thing)

		_, payload, err := c.segmentDecrypter.Decrypt(matches[4])
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to decode segment payload from source %v", this.Thing)
			return nil, twirp.NewError(twirp.Internal, "Failed to decode segment request payload")
		}

		weaverString := ""

		weaverBytes, err := json.Marshal(payload)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to convert weaver payload to JSON")
		} else {
			weaverString = string(weaverBytes)
		}

		return &rpc.Thing{
			Type:              rpc.Thing_TwitchSegmentURL,
			EdgeNode:          matches[1],
			EdgeCluster:       matches[2],
			AbsType:           matches[3],
			MuxType:           matches[5],
			TwitchChannel:     payload.Session.AnalyticsData.ChannelName,
			TwitchChannelId:   payload.Session.AnalyticsData.ContentId,
			WeaverPayloadJson: weaverString,
		}, nil
	}

	if ivsSegmentURLRegex.MatchString(this.Thing) {
		matches := ivsSegmentURLRegex.FindStringSubmatch(this.Thing)

		_, payload, err := c.segmentDecrypter.Decrypt(matches[3])
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to decode segment payload from source %v", this.Thing)
			return nil, twirp.NewError(twirp.Internal, "Failed to decode segment request payload")
		}

		// aws.ivs.us-west-2.298083573632.channel.hdviye1zVPxT
		rawChannel := strings.Split(payload.Session.AssignmentRequest.Stream.Channel, ".")

		weaverString := ""

		weaverBytes, err := json.Marshal(payload)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to convert weaver payload to JSON")
		} else {
			weaverString = string(weaverBytes)
		}

		return &rpc.Thing{
			Type:              rpc.Thing_IVSSegmentURL,
			EdgeNode:          matches[1],
			EdgeCluster:       matches[2],
			MuxType:           matches[4],
			IvsRegion:         rawChannel[2],
			IvsCustomerId:     rawChannel[3],
			IvsChannelId:      rawChannel[5],
			WeaverPayloadJson: weaverString,
		}, nil
	}

	if twitchStreamKeyRegex.MatchString(this.Thing) {
		matches := twitchStreamKeyRegex.FindStringSubmatch(this.Thing)

		return &rpc.Thing{
			Type:            rpc.Thing_TwitchStreamKey,
			TwitchChannelId: matches[1],
		}, nil
	}

	if ivsStreamKeyRegex.MatchString(this.Thing) {
		matches := ivsStreamKeyRegex.FindStringSubmatch(this.Thing)

		return c.getIVSStreamKey(ctx, matches[1], this.Thing)
	}

	if ivsChannelArn.MatchString(this.Thing) {
		matches := ivsChannelArn.FindStringSubmatch(this.Thing)

		return &rpc.Thing{
			Type:          rpc.Thing_IVSChannelARN,
			IvsRegion:     matches[1],
			IvsCustomerId: matches[2],
			IvsChannelId:  matches[3],
		}, nil
	}

	if h264ProfileLevelHex.MatchString(this.Thing) {
		result, err := profilehex.ParseHexString(this.Thing)

		if err != nil {
			c.logger.WithError(err).Warnf("Failed to decode avc profile hex from source %v", this.Thing)
			return nil, twirp.NewError(twirp.Internal, "Failed to decode profile hex")
		}

		return &rpc.Thing{
			Type:            rpc.Thing_AVCProfileHex,
			AvcProfileName:  result.Profile,
			AvcProfileLevel: result.Level,
		}, nil
	}

	if thing := getGenericThing(this.Thing); thing != nil {
		return thing, nil
	}

	return nil, twirp.NewError(twirp.Malformed, "Unable to determine input type")
}
