import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import ecr = require('@aws-cdk/aws-ecr');
import iam = require('@aws-cdk/aws-iam');
import sqs = require('@aws-cdk/aws-sqs');
import ddb = require('@aws-cdk/aws-dynamodb');
import lambda = require('@aws-cdk/aws-lambda');
import * as lambdaEvents from '@aws-cdk/aws-lambda-event-sources'
import s3 = require('@aws-cdk/aws-s3');

export interface CarrotRtmpRecorderAPIProps {
    invokeFunctionRoles: string[]
    recordingEndpointRegions: string[]
    digestionEndpoints: string[]
    carrotStreamAnalysisLambdaARN: string
    rtmpsDomain: string
    flvAnalyserRecordingBucketName: string
    dgnARNs: string[]
}

export class CarrotRtmpRecorderAPI extends cdk.Construct {
    Repository: ecr.Repository
    EndpointsTable: ddb.Table
    DumpsTable: ddb.Table
    ServiceLambda: lambda.Function
    RTMPDumpDLQ: sqs.Queue
    RTMPDumpQueue: sqs.Queue
    RTMPDumpLambda: lambda.Function

    constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CarrotRtmpRecorderAPIProps) {
        super(scope, id);

        this.Repository = new ecr.Repository(this, 'ECRRepository', {
            repositoryName: 'carrot-rtmp-recorder',
        })

        // Dynamo db table for tracking RTMP capture endpoints
        this.EndpointsTable = new ddb.Table(this, 'EndopintsTable', {
            tableName: 'carrot-rtmp-capture-endpoints',
            partitionKey: {name: 'id', type: ddb.AttributeType.STRING},
            billingMode: ddb.BillingMode.PAY_PER_REQUEST,
            timeToLiveAttribute: 'ttl',
        })

        this.EndpointsTable.addGlobalSecondaryIndex({
            indexName: 'idx-owner-by-created-at',
            partitionKey: {name: 'endpoint_owner', type: ddb.AttributeType.STRING},
            projectionType: ddb.ProjectionType.ALL,
            sortKey: {name: 'created_at', type: ddb.AttributeType.STRING},
        })

        this.EndpointsTable.addGlobalSecondaryIndex({
            indexName: 'idx-region',
            partitionKey: {name: 'region', type: ddb.AttributeType.STRING},
            projectionType: ddb.ProjectionType.ALL,
        })

        // Dynamo db table for tracking RTMP dumps
        this.DumpsTable = new ddb.Table(this, 'DumpsTable', {
            tableName: 'carrot-rtmp-dumps',
            partitionKey: {name: 'id', type: ddb.AttributeType.STRING},
            billingMode: ddb.BillingMode.PAY_PER_REQUEST,
            timeToLiveAttribute: 'ttl',
        })

        this.DumpsTable.addGlobalSecondaryIndex({
            indexName: 'idx-owner-by-created-at',
            partitionKey: {name: 'dump_owner', type: ddb.AttributeType.STRING},
            projectionType: ddb.ProjectionType.ALL,
            sortKey: {name: 'created_at', type: ddb.AttributeType.STRING},
        })

        // We should add a lambda to listen to these and mark them as failed
        this.RTMPDumpDLQ = new sqs.Queue(this, 'RTMPDumpDLQ', {})

        this.RTMPDumpQueue = new sqs.Queue(this, 'RTMPDumpQueue', {
            queueName: 'rtmp-dump-queue',
            deadLetterQueue: {
                maxReceiveCount: 3,
                queue: this.RTMPDumpDLQ,
            },
            visibilityTimeout: cdk.Duration.minutes(3), // This matches the rtmp dump lambda timeout in carrot-rtmp-recorder
        })

        // Lambda for the service API
        this.ServiceLambda = new lambda.Function(this, 'RecorderService', {
            vpc: vpc,
            runtime: lambda.Runtime.GO_1_X,
            memorySize: 1024,
            timeout: cdk.Duration.seconds(30),
            handler: 'bin/carrot-rtmp-recorder-api',
            code: lambda.Code.fromAsset('../bin/carrot-rtmp-recorder-api.zip'),
            environment: {
                "endpointsTableName": this.EndpointsTable.tableName,
                "rtmpDumpsTableName": this.DumpsTable.tableName,
                "rtmpDumpQueueURL": this.RTMPDumpQueue.queueUrl,
                "availableRegions": props.recordingEndpointRegions.join(','),
                "digestionEndpoints": props.digestionEndpoints.join(','),
                "csaLambdaARN": props.carrotStreamAnalysisLambdaARN,
                "rtmpsDomain": props.rtmpsDomain,
            }
        })

        // Allow things to call the function
        for (let roleArn of props.invokeFunctionRoles) {
            this.ServiceLambda.grantInvoke(new iam.ArnPrincipal(roleArn))
        }

        this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
            sid: 'DDBAccess',
            effect: iam.Effect.ALLOW,
            actions: [
                "dynamodb:PutItem",
                "dynamodb:UpdateItem",
                "dynamodb:GetItem",
                "dynamodb:Scan",
                "dynamodb:Query"
            ],
            resources: [
                this.EndpointsTable.tableArn,
                this.EndpointsTable.tableArn + '/*',
                this.DumpsTable.tableArn,
                this.DumpsTable.tableArn + '/*',
            ]
        }))

        this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
            sid: 'SQSAccess',
            effect: iam.Effect.ALLOW,
            actions: [
                "sqs:SendMessage",
            ],
            resources: [
                this.RTMPDumpQueue.queueArn,
            ]
        }))

        // Grant the Lambda Role access to call api-execute on IOCP
        this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                "execute-api:Invoke",
            ],
            resources: props.dgnARNs
        }))

        /*====================
          RTMP Dump Lambda
        =====================*/
        const recordingBucketRef = s3.Bucket.fromBucketName(this, 'RecordingBucketRef', props.flvAnalyserRecordingBucketName)

        this.RTMPDumpLambda = new lambda.Function(this, 'RTMPDumpLambda', {
            vpc: vpc,
            runtime: lambda.Runtime.GO_1_X,
            memorySize: 512,
            timeout: cdk.Duration.minutes(3), // We're setting max RTMP dump duration to 2 mins so give us some wiggle room
            handler: 'bin/carrot-rtmp-dumper',
            code: lambda.Code.asset('../bin/carrot-rtmp-dumper.zip'),
            environment: {
                "recordingAnalysisBucketName": recordingBucketRef.bucketName,
                "rtmpCaptureEndpointsTableName": this.EndpointsTable.tableName,
                "rtmpDumpsTableName": this.DumpsTable.tableName,
            },
            reservedConcurrentExecutions: 10, // This is essentially going to be the max number of queries running on the cluster at the same time
        })

        // Executor needs to update the status of the query on completion
        this.RTMPDumpLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                "dynamodb:UpdateItem",
            ],
            resources: [
                this.DumpsTable.tableArn,
                this.DumpsTable.tableArn + '/*',
            ]
        }));

        // Allow the function to write the dump to S3
        this.RTMPDumpLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                "s3:PutObject",
            ],
            resources: [
                recordingBucketRef.arnForObjects("*"),
            ]
        }))

        // Link it up to the SQS queue
        this.RTMPDumpLambda.addEventSource(new lambdaEvents.SqsEventSource(this.RTMPDumpQueue, {
            batchSize: 1,
        }))
    }
}
