import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import ecr = require('@aws-cdk/aws-ecr');
import ecs = require('@aws-cdk/aws-ecs');
import s3 = require('@aws-cdk/aws-s3');
import acm = require('@aws-cdk/aws-certificatemanager');
import route53 = require('@aws-cdk/aws-route53');
import elb = require('@aws-cdk/aws-elasticloadbalancingv2');
import dynamodb = require('@aws-cdk/aws-dynamodb');
import iam = require('@aws-cdk/aws-iam');
import route53targets = require('@aws-cdk/aws-route53-targets');
import {TargetType} from "@aws-cdk/aws-elasticloadbalancingv2";
import {Protocol} from "@aws-cdk/aws-ecs";

export interface CarrotRTMPRecorderProps {
    ecrRepositoryArn: string
    ecrRepositoryName: string
    controlAPIRegion: string
    endpointTableArn: string
    recordingBucketName: string
    resourceGroupName: string
    resourceGroupValue: string
    rtmpsHostName: string
    hostedZoneName: string
}

export class CarrotRTMPRecorder extends cdk.Construct {
    TaskDefinition: ecs.TaskDefinition
    NLB: elb.NetworkLoadBalancer

    constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CarrotRTMPRecorderProps) {
        super(scope, id);

        const recordingBucketRef = s3.Bucket.fromBucketName(this, 'RecordingBucketRef', props.recordingBucketName)
        const endpointTableRef = dynamodb.Table.fromTableArn(this, 'EndpointTableRef', props.endpointTableArn)

        const cluster = new ecs.Cluster(this, 'RecorderCluster', {
            vpc: vpc,
        })

        cdk.Tags.of(cluster).add(props.resourceGroupName, props.resourceGroupValue);

        // This is assigned when we start the task but we're defining it here
        const allowRtmp = new ec2.SecurityGroup(this, 'AllowAllSecurityGroup', {
            vpc: vpc,
        })

        allowRtmp.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(1935), 'Allow rtmp connections from anywhere')

        cdk.Tags.of(allowRtmp).add(props.resourceGroupName, props.resourceGroupValue);

        this.TaskDefinition = new ecs.FargateTaskDefinition(this, 'RecorderTaskDef', {
            memoryLimitMiB: 1024,
        });

        cdk.Tags.of(this.TaskDefinition).add(props.resourceGroupName, props.resourceGroupValue);

        // Allow writing to S3
        this.TaskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                "s3:AbortMultipartUpload",
                "s3:CompleteMultipartUpload",
                "s3:CreateMultipartUpload",
                "s3:PutObject",
                "s3:UploadPart"
            ],
            resources: [
                recordingBucketRef.bucketArn,
                recordingBucketRef.arnForObjects('*'),
            ],
        }))

        // Allow updating status in dynamodb
        this.TaskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                "dynamodb:UpdateItem",
                "dynamodb:Query"
            ],
            resources: [
                endpointTableRef.tableArn,
                endpointTableRef.tableArn + '/*',
            ],
        }))

        // Allow stopping of tasks
        this.TaskDefinition.addContainer('RecorderContainer', {
            image: ecs.ContainerImage.fromEcrRepository(ecr.Repository.fromRepositoryAttributes(this, 'ECRRepoRef', {
                repositoryArn: props.ecrRepositoryArn,
                repositoryName: props.ecrRepositoryName,
            })),
            memoryLimitMiB: 1024,
            logging: ecs.LogDrivers.awsLogs({
                streamPrefix: 'carrot-rtmp-recorder',
            }),
            stopTimeout: cdk.Duration.seconds(30),
            healthCheck: {
                command: ['CMD-SHELL', 'curl -f http://localhost:1935/ || exit 1'],
                interval: cdk.Duration.seconds(5),
                timeout: cdk.Duration.seconds(2),
                startPeriod: cdk.Duration.seconds(10),
                retries: 3,
            },
            environment: {
                CRR_controlAPIRegion: props.controlAPIRegion,
                CRR_endpointTableName: endpointTableRef.tableName,
                CRR_destBucketName: recordingBucketRef.bucketName,
            },
            portMappings: [{
                containerPort: 1935,
                hostPort: 1935,
                protocol: Protocol.TCP,
            }]
        });

        const service = new ecs.FargateService(this, 'Service', {
            cluster,
            securityGroups: [allowRtmp],
            assignPublicIp: false,
            taskDefinition: this.TaskDefinition,
            desiredCount: 1
        });

        // Create NLB
        this.NLB = new elb.NetworkLoadBalancer(this, 'RTMPNLB', {
            vpc: vpc,
            crossZoneEnabled: true,
            internetFacing: true,
            loadBalancerName: 'carrot-rtmp-recorder-nlb',
        })

        cdk.Tags.of(this.NLB).add(props.resourceGroupName, props.resourceGroupValue);

        const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZoneRef', {
            domainName: props.hostedZoneName,
        })

        const rtmpsCert = new acm.DnsValidatedCertificate(this, 'RTMPTLSCert', {
            domainName: props.rtmpsHostName,
            hostedZone: hostedZone,
        });

        cdk.Tags.of(rtmpsCert).add(props.resourceGroupName, props.resourceGroupValue);

        const rtmpTargetGroup = new elb.NetworkTargetGroup(this, 'RTMPSTargetGroup', {
            vpc: vpc,
            port: 443,
            targetType: TargetType.IP,
        })

        service.attachToNetworkTargetGroup(rtmpTargetGroup)

        this.NLB.addListener('RTMPListener', {
            port: 443,
            protocol: elb.Protocol.TLS,
            certificates: [
                elb.ListenerCertificate.fromCertificateManager(rtmpsCert)
            ],
            defaultTargetGroups: [rtmpTargetGroup],
        })

        // Create DNS entries for the RTMPS NLB
        new route53.ARecord(this, 'RtmpsARecord', {
            zone: hostedZone,
            recordName: props.rtmpsHostName,
            target: route53.RecordTarget.fromAlias(new route53targets.LoadBalancerTarget(this.NLB))
        })
    }
}
