package main

import (
	"fmt"
	"os"
	"strings"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	rpc "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/svc"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/sirupsen/logrus"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	availableRegions := strings.Split(os.Getenv("availableRegions"), ",")
	digestionEndpoints := strings.Split(os.Getenv("digestionEndpoints"), ",")
	endpointsTableName := os.Getenv("endpointsTableName")
	rtmpsDomain := os.Getenv("rtmpsDomain")
	rtmpDumpsTableName := os.Getenv("rtmpDumpsTableName")
	rtmpDumpQueueURL := os.Getenv("rtmpDumpQueueURL")

	// Create am AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	ddb := dynamodb.New(sess)
	s3 := s3.New(sess)
	sqs := sqs.New(sess)
	lambdaClient := lambda.New(sess)

	// Create the Carrot Stream Analysis client
	csaLambdaARN := os.Getenv("csaLambdaARN")
	lambdaTransport := twirpLambda.NewClient(lambdaClient, csaLambdaARN)
	csaClient := csa.NewCarrotStreamAnalysisProtobufClient("https://would.you.like.a.carrot.twitch.tv", lambdaTransport)

	// Create the Service
	service := svc.New(
		availableRegions,
		digestionEndpoints,
		endpointsTableName,
		rtmpDumpsTableName,
		rtmpDumpQueueURL,
		rtmpsDomain,
		ddb,
		s3,
		sqs,
		csaClient,
		logger)

	// Create a Twirp service based off the internal service
	err = twirpLambda.ListenAndServe(":80", rpc.NewCarrotRtmpRecorderServer(service, nil))

	if err != nil {
		panic(err)
	}
}
