package main

import (
	"os"
	"os/signal"
	"syscall"
	"time"

	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/server"
	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/svc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/sirupsen/logrus"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.InfoLevel)

	controlAPIRegion := os.Getenv("CRR_controlAPIRegion")
	endpointTableName := os.Getenv("CRR_endpointTableName")
	destBucketName := os.Getenv("CRR_destBucketName")
	awsRegion := os.Getenv("AWS_REGION")

	if controlAPIRegion == "" {
		logger.Error("Missing environment variable: CRR_controlAPIRegion")
		os.Exit(1)
	}

	if endpointTableName == "" {
		logger.Error("Missing environment variable: CRR_endpointTableName")
		os.Exit(1)
	}

	if destBucketName == "" {
		logger.Error("Missing environment variable: CRR_destBucketName")
		os.Exit(1)
	}

	controlAPISession, err := session.NewSession(&aws.Config{
		Region: aws.String(controlAPIRegion),
	})

	if err != nil {
		logger.WithError(err).Error("Failed to generate AWS session for s3 client")
		os.Exit(1)
	}

	s3Client := s3.New(controlAPISession)
	dynamoDbClient := dynamodb.New(controlAPISession)

	operations := svc.NewWorkerOperations(endpointTableName, "", dynamoDbClient, logger)

	server := server.New(destBucketName, awsRegion, s3Client, operations, logger)

	err = server.Start()
	if err != nil {
		logger.WithError(err).Error("Unable to start RTMP server")
		os.Exit(1)
	}

	logger.Info("Server started listening")

	serverStopped := make(chan interface{}, 1)

	go func() {
		for {
			if server.Stopped() {
				serverStopped <- "would you like a carrot?"
				return
			}

			time.Sleep(1000)
		}
	}()

	osSig := make(chan os.Signal, 1)
	signal.Notify(osSig, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)

	select {
	case sig := <-osSig:
		logger.Infof("Got signal %v, cleaning up", sig)
		server.Stop()
	case <-serverStopped:
		logger.Info("Server Stopped itself, exiting")
	}

	logger.Info("Cleaned up, exiting")
}
