package main

import (
	"context"
	"encoding/json"
	"fmt"
	"os"

	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rtmpdump"
	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/svc"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/sirupsen/logrus"
)

var (
	rtmpDumper rtmpdump.Dumper
	logger     *logrus.Logger
)

func init() {
	logger = logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	recordingAnalysisBucketName := os.Getenv("recordingAnalysisBucketName")
	rtmpDumpsTableName := os.Getenv("rtmpDumpsTableName")

	// We don't really need this but I'm passing it in anyway because I dont want to feel dirty
	rtmpCaptureEndpointsTableName := os.Getenv("rtmpCaptureEndpointsTableName")

	// Create am AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(os.Getenv("AWS_REGION")),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	s3Client := s3.New(sess)
	ddbClient := dynamodb.New(sess)
	operations := svc.NewWorkerOperations(rtmpCaptureEndpointsTableName, rtmpDumpsTableName, ddbClient, logger)

	rtmpDumper = rtmpdump.New(s3Client, recordingAnalysisBucketName, operations, logger)
}

func main() {
	lambda.Start(handle)
}

func handle(ctx context.Context, sqsEvent events.SQSEvent) error {
	// There should only be 1 of these as batchSize is set to 1
	if len(sqsEvent.Records) > 1 {
		return fmt.Errorf("Expected 1 record, got %v", len(sqsEvent.Records))
	}

	for _, message := range sqsEvent.Records {
		var querydumpReq svc.RTMPDumpRequest

		err := json.Unmarshal([]byte(message.Body), &querydumpReq)

		if err != nil {
			return err
		}

		err = rtmpDumper.DumpToS3(querydumpReq)

		if err != nil {
			return err
		}
	}

	return nil
}
