package rtmpdump

import (
	"context"
	"time"

	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/ms2s3"
	rpc "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/svc"
	rtmplog "code.justin.tv/event-engineering/gortmp/pkg/log"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	player "code.justin.tv/event-engineering/rtmp/pkg/player"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/sirupsen/logrus"
)

type Dumper interface {
	DumpToS3(dumpRequest svc.RTMPDumpRequest) error
}

type dumper struct {
	logger     logrus.FieldLogger
	s3         s3iface.S3API
	operations svc.WorkerOperations
	s3Bucket   string
}

func New(s3Client s3iface.S3API, s3Bucket string, operations svc.WorkerOperations, logger logrus.FieldLogger) Dumper {
	return &dumper{
		logger:     logger,
		s3:         s3Client,
		operations: operations,
		s3Bucket:   s3Bucket,
	}
}

func (d *dumper) DumpToS3(dumpRequest svc.RTMPDumpRequest) error {
	errCh := make(chan error)

	rtmplog.SetLogLevel(rtmplog.LogError)
	player := player.NewPlayer(errCh, d.logger)

	ctx := context.Background()
	ms := gortmp.NewMediaStream(ctx, "Would you like a carrot?")
	err := player.Play(dumpRequest.RTMPURL, ms)

	if err != nil {
		d.operations.UpdateRTMPDumpStatus(dumpRequest.RTMPDumpID, rpc.RTMPDumpStatus_DumpFailed)
		return err
	}

	recorder := ms2s3.New(d.s3, d.s3Bucket, d.logger)

	go func() {
		plErr := <-errCh
		d.logger.WithError(plErr).Warn("Error playing stream, stopping rtmpdump")
		player.Close()
		recorder.Stop()
		d.operations.UpdateRTMPDumpStatus(dumpRequest.RTMPDumpID, rpc.RTMPDumpStatus_DumpFailed)
	}()

	err = recorder.RecordToS3(ctx, ms, dumpRequest.S3Key)

	if err != nil {
		d.operations.UpdateRTMPDumpStatus(dumpRequest.RTMPDumpID, rpc.RTMPDumpStatus_DumpFailed)
		return err
	}

	d.operations.UpdateRTMPDumpStatus(dumpRequest.RTMPDumpID, rpc.RTMPDumpStatus_DumpInProgress)

	waitTime := time.NewTimer(time.Second * time.Duration(dumpRequest.DurationSeconds)).C

	<-waitTime

	player.Close()
	ms.Close()

	recorder.WaitUntilDone()

	d.operations.UpdateRTMPDumpStatus(dumpRequest.RTMPDumpID, rpc.RTMPDumpStatus_DumpComplete)

	return nil
}
