package svc

import (
	"context"
	"errors"
	"fmt"
	"time"

	crr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/google/uuid"
	"google.golang.org/protobuf/types/known/timestamppb"
)

// 48 hours is the max amount of time that an endpoint can exist for and 7 days is the amount of time that captured files last for
var endpointTTL = time.Hour * 48
var s3Expiry = time.Hour * 24 * 7

const maxMaxDurationSeconds = 21600 // 6 Hours
const rtmpsUrlFormat = "rtmps://%s.%s:443/app/%s"

func (c *client) GetAvailableCaptureEndpointRegions(_ context.Context, _ *crr.GetAvailableCaptureEndpointRegionsRequest) (*crr.AvailableCaptureEndpointRegions, error) {
	return &crr.AvailableCaptureEndpointRegions{
		Regions: c.availableRegions,
	}, nil
}

func (c *client) CreateCaptureEndpoint(ctx context.Context, request *crr.CreateCaptureEndpointRequest) (*crr.CaptureEndpointInfo, error) {
	// Check to make sure the region requested is actually one we can use
	if !c.regionIsValid(request.Region) {
		return nil, fmt.Errorf("invalid Region: %v", request.Region)
	}

	if request.Owner == "" {
		return nil, fmt.Errorf("please supply an owner")
	}

	if request.MaxDurationSeconds > maxMaxDurationSeconds {
		return nil, fmt.Errorf("max Duration cannot be greater than %v", maxMaxDurationSeconds)
	}

	if request.MaxDurationSeconds == 0 {
		request.MaxDurationSeconds = 600
	}

	endpointID := uuid.New().String()

	// Create the entry in dynamo
	endpoint := &ddbCaptureEndpoint{
		Owner:           request.Owner,
		Region:          request.Region,
		ID:              endpointID,
		Name:            request.Name,
		CreatedAt:       time.Now(),
		KeyPrefix:       fmt.Sprintf("captures/%v/", endpointID),
		StreamKey:       uuid.New().String(),
		Status:          crr.EndpointStatus_EndpointInitialised,
		EndpointExpires: time.Now().Add(endpointTTL).Unix(),               // When should the endpoint no longer function
		TTL:             time.Now().Add(endpointTTL).Add(s3Expiry).Unix(), // Make sure the record is in the DB for as long as it needs to be
		MaxDuration:     request.MaxDurationSeconds,                       // Max duration of the capture
	}

	item, err := dynamodbattribute.MarshalMap(endpoint)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal ddbCaptureEndpoint")
		return nil, errors.New("failed to create endpoint")
	}

	_, err = c.ddb.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		TableName: aws.String(c.endpointsTableName),
		Item:      item,
	})

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to insert ddbCaptureEndpoint into database")
		return nil, errors.New("failed to create endpoint")
	}

	return &crr.CaptureEndpointInfo{
		Id:     endpoint.ID,
		Region: endpoint.Region,
		Name:   endpoint.Name,
	}, nil
}

func (c *client) GetCaptureEndpoint(ctx context.Context, request *crr.GetCaptureEndpointRequest) (*crr.CaptureEndpointDetail, error) {
	idAttr, err := dynamodbattribute.Marshal(request.Id)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to get capture endpoint: Marshal ID")
		return nil, errors.New("failed to retrieve endpoint")
	}

	resp, err := c.ddb.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(c.endpointsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"id": idAttr,
		},
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to get item from database")
		return nil, errors.New("failed to retrieve endpoint")
	}

	var ddbce ddbCaptureEndpoint
	err = dynamodbattribute.UnmarshalMap(resp.Item, &ddbce)

	if err != nil {
		c.logger.WithError(err).Warn("Failed to convert item to struct")
		return nil, errors.New("failed to retrieve endpoint")
	}

	capturedFiles, err := c.csa.ListCapturedFiles(ctx, &csa.ListCapturedFilesRequest{
		Prefix: ddbce.KeyPrefix,
	})

	var files []*csa.CapturedFileInfo

	if err != nil {
		c.logger.WithError(err).Warnf("failed to retrieve captured file list for capture endpoint %v", ddbce.ID)
	} else {
		files = capturedFiles.Files
	}

	return &crr.CaptureEndpointDetail{
		Info: &crr.CaptureEndpointInfo{
			Region:  ddbce.Region,
			Id:      ddbce.ID,
			Name:    ddbce.Name,
			Status:  ddbce.Status,
			Expires: &timestamppb.Timestamp{Seconds: ddbce.EndpointExpires},
		},
		RtmpUrl: fmt.Sprintf(rtmpsUrlFormat, ddbce.Region, c.rtmpsDomain, ddbce.StreamKey),
		Files:   files,
	}, nil
}

func (c *client) ListCaptureEndpoints(ctx context.Context, request *crr.ListCaptureEndpointsRequest) (*crr.ListCaptureEndpointsResponse, error) {
	resp, err := c.ddb.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:              aws.String(c.endpointsTableName),
		IndexName:              aws.String("idx-owner-by-created-at"),
		KeyConditionExpression: aws.String("endpoint_owner = :endpoint_owner"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":endpoint_owner": {S: &request.Owner},
		},
		ScanIndexForward: aws.Bool(false),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to query endpoints table")
		return nil, errors.New("failed to retrieve endpoint list")
	}

	result := make([]*crr.CaptureEndpointInfo, 0, len(resp.Items))

	for _, item := range resp.Items {
		var ddbce ddbCaptureEndpoint
		err = dynamodbattribute.UnmarshalMap(item, &ddbce)

		if err != nil {
			c.logger.WithError(err).Warn("failed to convert item to struct")
			continue
		}

		result = append(result, &crr.CaptureEndpointInfo{
			Region:  ddbce.Region,
			Id:      ddbce.ID,
			Name:    ddbce.Name,
			Status:  ddbce.Status,
			Expires: &timestamppb.Timestamp{Seconds: ddbce.EndpointExpires},
		})
	}

	return &crr.ListCaptureEndpointsResponse{
		Endpoints: result,
	}, nil
}
