package svc

import (
	"fmt"
	"time"

	crr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/sirupsen/logrus"
)

type ddbCaptureEndpoint struct {
	Owner           string             `json:"endpoint_owner"`
	Region          string             `json:"region"`
	ID              string             `json:"id"`
	Name            string             `json:"name"`
	CreatedAt       time.Time          `json:"created_at"`
	KeyPrefix       string             `json:"key_prefix"` // For S3
	StreamKey       string             `json:"stream_key"`
	Status          crr.EndpointStatus `json:"status"`
	EndpointExpires int64              `json:"endpoint_expires"`
	TTL             int64              `json:"ttl"`
	MaxDuration     uint64             `json:"max_duration"`
}

type workerOperations struct {
	endpointsTableName string
	rtmpDumpsTableName string
	ddb                dynamodbiface.DynamoDBAPI
	logger             logrus.FieldLogger
}

type WorkerOperations interface {
	UpdateEndpointStatus(id string, status crr.EndpointStatus)
	UpdateRTMPDumpStatus(id string, status crr.RTMPDumpStatus)
	GetRegionEndpointConfig(region string) ([]*ddbCaptureEndpoint, error)
}

func NewWorkerOperations(endpointsTableName, rtmpDumpsTableName string, ddb dynamodbiface.DynamoDBAPI, logger logrus.FieldLogger) WorkerOperations {
	return &workerOperations{
		endpointsTableName: endpointsTableName,
		rtmpDumpsTableName: rtmpDumpsTableName,
		ddb:                ddb,
		logger:             logger,
	}
}

func (c *workerOperations) UpdateEndpointStatus(id string, status crr.EndpointStatus) {
	idAttr, err := dynamodbattribute.Marshal(id)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal ID")
	}

	statusAttr, err := dynamodbattribute.Marshal(status)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal Status")
	}

	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: aws.String(c.endpointsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"id": idAttr,
		},
		ExpressionAttributeNames: map[string]*string{
			"#s": aws.String("status"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":status": statusAttr,
			":id":     idAttr,
		},
		UpdateExpression:    aws.String("SET #s=:status"),
		ConditionExpression: aws.String("id=:id"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: UpdateItem")
	}
}

type ddbRTMPDump struct {
	Owner     string             `json:"dump_owner"`
	ID        string             `json:"id"`
	Name      string             `json:"name"`
	CreatedAt time.Time          `json:"created_at"`
	S3Key     string             `json:"s3_key"`
	Status    crr.RTMPDumpStatus `json:"status"`
	TTL       int64              `json:"ttl"`
}

func (c *workerOperations) UpdateRTMPDumpStatus(id string, status crr.RTMPDumpStatus) {
	idAttr, err := dynamodbattribute.Marshal(id)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal ID")
	}

	statusAttr, err := dynamodbattribute.Marshal(status)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal Status")
	}

	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: aws.String(c.rtmpDumpsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"id": idAttr,
		},
		ExpressionAttributeNames: map[string]*string{
			"#s": aws.String("status"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":status": statusAttr,
			":id":     idAttr,
		},
		UpdateExpression:    aws.String("SET #s=:status"),
		ConditionExpression: aws.String("id=:id"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to update rtmp dump status: UpdateItem")
	}
}

func (c *workerOperations) GetRegionEndpointConfig(region string) ([]*ddbCaptureEndpoint, error) {
	resp, err := c.ddb.Query(&dynamodb.QueryInput{
		TableName:              aws.String(c.endpointsTableName),
		IndexName:              aws.String("idx-region"),
		KeyConditionExpression: aws.String("#r = :config_region"),
		ExpressionAttributeNames: map[string]*string{
			"#r": aws.String("region"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":config_region": {S: &region},
		},
		ScanIndexForward: aws.Bool(false),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to query endpoints table")
		return nil, fmt.Errorf("failed to retrieve endpoint list")
	}

	result := make([]*ddbCaptureEndpoint, 0, len(resp.Items))

	for _, item := range resp.Items {
		var ddbce ddbCaptureEndpoint
		err = dynamodbattribute.UnmarshalMap(item, &ddbce)

		if err != nil {
			c.logger.WithError(err).Warn("failed to convert item to struct")
			continue
		}

		result = append(result, &ddbce)
	}

	return result, nil
}
