package svc

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"strings"
	"time"

	crr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	"code.justin.tv/video/digestion/pkg/dgn"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/golang/protobuf/ptypes"
	"github.com/google/uuid"
	"github.com/twitchtv/twirp"
)

// 7 days is the amount of time that RTMP dumps are kept
var rtmpDumpTTL = time.Hour * 24 * 7

type RTMPDumpRequest struct {
	RTMPDumpID      string `json:"rtmp_dump_id"`
	RTMPURL         string `json:"rtmp_url"`
	S3Key           string `json:"s3_key"`
	DurationSeconds uint32 `json:"duration_seconds"`
}

func (c *client) CreateRTMPDump(ctx context.Context, req *crr.CreateRTMPDumpRequest) (*crr.CreateRTMPDumpResponse, error) {
	if req.Owner == "" {
		return nil, twirp.InvalidArgumentError("Owner", "Empty")
	}

	if req.Region == "" {
		return nil, twirp.InvalidArgumentError("Region", "Empty")
	}

	if req.CustomerId == "" {
		return nil, twirp.InvalidArgumentError("CustomerId", "Empty")
	}

	if req.ChannelId == "" {
		return nil, twirp.InvalidArgumentError("ChannelId", "Empty")
	}

	channel := fmt.Sprintf("aws.ivs.%v.%v.channel.%v", req.Region, req.CustomerId, req.ChannelId)

	if req.CustomerId == "twitch" {
		channel = req.ChannelId
	}

	if req.Format == "" {
		req.Format = "live"
	}

	if req.DurationSeconds == 0 {
		req.DurationSeconds = 30
	}

	var rtmpURL string

	if req.CustomerId != "twitch" {
		// Check if the
	}

	for _, digestionURL := range c.digestionEndpoints {
		splitURL := strings.Split(digestionURL, ".")
		if len(splitURL) < 2 {
			return nil, twirp.InternalErrorWith(fmt.Errorf("digestion url doesn't seem to have a region: %s", digestionURL))
		}
		signerClient, err := getSignerClient(http.DefaultClient, splitURL[1])
		if err != nil {
			return nil, twirp.InternalErrorWith(fmt.Errorf("could not create digestion signer client: %w", err))
		}
		dgnClient := dgn.NewDigestionV1ProtobufClient(digestionURL, signerClient)

		dResp, err := dgnClient.GetChannel(ctx, &dgn.ChannelRequest{
			Channel: channel,
		})

		if err != nil {
			if twErr, ok := err.(twirp.Error); !ok || twErr.Code() != twirp.NotFound {
				c.logger.WithError(err).Warnf("Failed to communicate with digestion endpoint %v", digestionURL)
			}

			continue
		}

		for _, stream := range dResp.Streams {
			if stream.Format != req.Format {
				continue
			}

			rtmpURL = stream.RtmpUrl
			break
		}
	}

	if rtmpURL == "" {
		return nil, twirp.NotFoundError("Could not find stream")
	}

	rtmpDumpID := uuid.New().String()

	rtmpDumpReq := RTMPDumpRequest{
		RTMPDumpID:      rtmpDumpID,
		RTMPURL:         rtmpURL,
		S3Key:           fmt.Sprintf("dumps/%v_%v.flv", rtmpDumpID, req.Format),
		DurationSeconds: req.DurationSeconds,
	}

	// Send to SQS
	sqsPayload, err := json.Marshal(rtmpDumpReq)
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal SQS payload")
		return nil, twirp.InternalError("Failed to create RTMP dump")
	}

	c.sqs.SendMessage(&sqs.SendMessageInput{
		QueueUrl:    &c.rtmpDumpQueueURL,
		MessageBody: aws.String(string(sqsPayload)),
	})

	// Write to dynamodb
	ddbd := &ddbRTMPDump{
		Owner:     req.Owner,
		ID:        rtmpDumpID,
		Name:      req.Name,
		CreatedAt: time.Now(),
		S3Key:     rtmpDumpReq.S3Key,
		Status:    crr.RTMPDumpStatus_DumpRequested,
		TTL:       time.Now().Add(rtmpDumpTTL).Unix(),
	}

	item, err := dynamodbattribute.MarshalMap(ddbd)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal ddbRTMPDump")
		return nil, twirp.InternalError("Failed to create RTMP dump")
	}

	_, err = c.ddb.PutItem(&dynamodb.PutItemInput{
		TableName: aws.String(c.rtmpDumpsTableName),
		Item:      item,
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to write rtmp dump info to dynamodb")
		return nil, twirp.InternalError("Failed to create RTMP dump")
	}

	return &crr.CreateRTMPDumpResponse{
		RtmpDumpId: rtmpDumpID,
	}, nil
}

func (c *client) ListRTMPDumps(ctx context.Context, request *crr.ListRTMPDumpsRequest) (*crr.ListRTMPDumpsResponse, error) {
	resp, err := c.ddb.Query(&dynamodb.QueryInput{
		TableName:              aws.String(c.rtmpDumpsTableName),
		IndexName:              aws.String("idx-owner-by-created-at"),
		KeyConditionExpression: aws.String("dump_owner = :dump_owner"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":dump_owner": &dynamodb.AttributeValue{S: &request.Owner},
		},
		ScanIndexForward: aws.Bool(false),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to query rtmp dumps table")
		return nil, errors.New("Failed to retrieve rtmp dump list")
	}

	result := make([]*crr.RTMPDumpSummary, 0, len(resp.Items))

	for _, item := range resp.Items {
		var ddbd ddbRTMPDump
		err = dynamodbattribute.ConvertFromMap(item, &ddbd)

		if err != nil {
			c.logger.WithError(err).Warn("Failed to convert item to struct")
			continue
		}

		createdAtProto, err := ptypes.TimestampProto(ddbd.CreatedAt)
		if err != nil {
			c.logger.WithError(err).Warn("Failed to convert created_at to protobuf timestamp")
		}

		result = append(result, &crr.RTMPDumpSummary{
			RtmpDumpId: ddbd.ID,
			Name:       ddbd.Name,
			CreatedAt:  createdAtProto,
			FilePath:   ddbd.S3Key,
			Status:     ddbd.Status,
		})
	}

	return &crr.ListRTMPDumpsResponse{
		RtmpDumps: result,
	}, nil
}

func (c *client) GetRTMPDump(ctx context.Context, request *crr.GetRTMPDumpRequest) (*crr.RTMPDumpSummary, error) {
	idAttr, err := dynamodbattribute.Marshal(request.RtmpDumpId)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to get RTMP dump: Marshal ID")
		return nil, errors.New("Failed to retrieve endpoint")
	}

	resp, err := c.ddb.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(c.rtmpDumpsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"id": idAttr,
		},
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to get item from database")
		return nil, errors.New("Failed to retrieve endpoint")
	}

	var ddbd ddbRTMPDump
	err = dynamodbattribute.ConvertFromMap(resp.Item, &ddbd)

	if err != nil {
		c.logger.WithError(err).Warn("Failed to convert item to struct")
		return nil, errors.New("Failed to retrieve endpoint")
	}

	createdAtProto, err := ptypes.TimestampProto(ddbd.CreatedAt)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to convert created_at to protobuf timestamp")
	}

	return &crr.RTMPDumpSummary{
		RtmpDumpId: ddbd.ID,
		Name:       ddbd.Name,
		CreatedAt:  createdAtProto,
		FilePath:   ddbd.S3Key,
		Status:     ddbd.Status,
	}, nil
}
