package svc

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	v4 "github.com/aws/aws-sdk-go/aws/signer/v4"
	"github.com/twitchtv/twirp"
)

const amznErrorTypeHeader = "x-amzn-ErrorType"
const amznAPIGatewayIDHeader = "x-amz-apigw-id"

func getSignerClient(httpClient *http.Client, region string) (*SignerClient, error) {
	// Strip off default redirect handler
	// https://github.com/twitchtv/twirp/blob/v5.9.0/protoc-gen-twirp/generator.go#L664
	httpClient.CheckRedirect = func(_ *http.Request, via []*http.Request) error {
		return http.ErrUseLastResponse
	}
	sess, err := session.NewSession(&aws.Config{Region: aws.String(region)})
	if err != nil {
		return nil, err
	}
	return &SignerClient{
		BaseClient: httpClient,
		Signer:     v4.NewSigner(sess.Config.Credentials),
		Service:    "execute-api",
		Region:     region,
	}, nil
}

type HTTPClient interface {
	Do(req *http.Request) (*http.Response, error)
}

type SignerClient struct {
	BaseClient HTTPClient
	Signer     *v4.Signer
	Service    string
	Region     string
}

func (s *SignerClient) Do(req *http.Request) (*http.Response, error) {
	b, err := ioutil.ReadAll(req.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to request read body for sigv4 signing: %s", err)
	}
	_, err = s.Signer.Sign(req, bytes.NewReader(b), s.Service, s.Region, time.Now())
	if err != nil {
		return nil, fmt.Errorf("failed to sigv4 sign the request: %s", err)
	}
	resp, err := s.BaseClient.Do(req)
	if err != nil {
		return resp, err
	}
	if amznErrType := resp.Header.Get(amznErrorTypeHeader); amznErrType != "" {
		if resp.Header.Get(amznAPIGatewayIDHeader) != "" {
			// API Gateway stopped us, upgrade 400s to 500s.
			defer resp.Body.Close()
			bodyBytes, err := ioutil.ReadAll(resp.Body)
			if err != nil {
				return nil, twirp.InternalError(fmt.Sprintf("error while reading api gateway error body: %v", err))
			}
			return nil, twirp.InternalError(fmt.Sprintf("got api gateway error type %s, error body: %s",
				amznErrType, string(bodyBytes)))
		}
	}
	return resp, err
}
