import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import lambda = require('@aws-cdk/aws-lambda');
import s3 = require('@aws-cdk/aws-s3');
import { S3EventSource } from '@aws-cdk/aws-lambda-event-sources';

export interface CarrotFLVAnalyserProps {
  recordingAnalysisBucketName: string
}

export class CarrotFLVAnalyser extends cdk.Construct {
  RecordingAnalysisBucket: s3.Bucket
  AnalyserLambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CarrotFLVAnalyserProps) {
    super(scope, id);

    this.RecordingAnalysisBucket = new s3.Bucket(this, 'RecordingAnalysisBucket', {
      bucketName: props.recordingAnalysisBucketName,
    })

    this.RecordingAnalysisBucket.addLifecycleRule({
      enabled: true,
      expiration: cdk.Duration.hours(24 * 7),
    })

    const mediaInfoLayer = new lambda.LayerVersion(this, 'MediaInfoLayer', {
      code: lambda.Code.asset('./layers/MediaInfo_CLI_20.03_Lambda.zip')
    })

    // Lambda for the analyser
    this.AnalyserLambda = new lambda.Function(this, 'Analyser', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 2048,
      layers: [mediaInfoLayer],
      timeout: cdk.Duration.minutes(15),
      handler: 'bin/linux/carrot-flv-analyser',
      code: lambda.Code.asset('../bin/carrot-flv-analyser.zip'),
      environment: {
        "recordingAnalysisBucketName": this.RecordingAnalysisBucket.bucketName,
      }
    })

    this.AnalyserLambda.addEventSource(new S3EventSource(this.RecordingAnalysisBucket, {
      events: [s3.EventType.OBJECT_CREATED],
      filters: [{ suffix: '.flv' }]
    }))

    this.AnalyserLambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'S3Stuff',
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:GetObject",
        "s3:PutObject",
      ],
      resources: [
        this.RecordingAnalysisBucket.arnForObjects("*")
      ]
    }))

    this.AnalyserLambda.addPermission('S3InvokePermissions', {
      action: "lambda:InvokeFunction",
      principal: new iam.ServicePrincipal('s3.amazonaws.com'),
      sourceArn: this.RecordingAnalysisBucket.bucketArn,
      sourceAccount: cdk.Stack.of(this).account,
    })
  }
}
