import lambda = require('@aws-cdk/aws-lambda');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import s3 = require('@aws-cdk/aws-s3');
import cdk = require('@aws-cdk/core');

export interface CSAProps {
  streamlogEndpoint: string
  flvAnalyserBucketName: string
  invokeFunctionRoles: string[]
}

export class CarrotStreamAnalysis extends cdk.Construct {
  CSALambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CSAProps) {
    super(scope, id);

    this.CSALambda = new lambda.Function(this, 'CarrotStreamAnalysis', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(10),
      handler: 'bin/linux/carrot-stream-analysis',
      code: lambda.Code.asset('../bin/carrot-stream-analysis.zip'),
      environment: {
        "streamlogEndpoint": props.streamlogEndpoint,
        "flvAnalyserBucketName": props.flvAnalyserBucketName,
      }
    })

    let flvAnalyserBucketArn = s3.Bucket.fromBucketName(this, 'RecordingAnalysisBucketRef', props.flvAnalyserBucketName)

    this.CSALambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'AllowS3Access',
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:HeadObject",
        "s3:GetObject",
        "s3:ListBucket",
      ],
      resources: [
        flvAnalyserBucketArn.bucketArn,
        flvAnalyserBucketArn.arnForObjects("*"),
      ]
    }))


    for (let roleArn of props.invokeFunctionRoles) {
      this.CSALambda.grantInvoke(new iam.ArnPrincipal(roleArn))
    }
  }
}
