package main

import (
	"context"
	"fmt"
	"os"
	"strings"

	mi "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/mediainfo"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/sirupsen/logrus"
)

var (
	mediaInfo           mi.MediaInfo
	logger              *logrus.Logger
	recordingBucketName string
)

func init() {
	logger = logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	recordingBucketName = os.Getenv("recordingAnalysisBucketName")

	// Create am AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	s3Client := s3.New(sess)

	mediaInfo = mi.New(recordingBucketName, s3Client, logger)
}

func main() {
	lambda.Start(handle)
}

func handle(ctx context.Context, event events.S3Event) {
	for _, record := range event.Records {
		s3 := record.S3

		if !strings.HasSuffix(s3.Object.Key, ".flv") {
			return
		}

		if recordingBucketName != s3.Bucket.Name {
			logger.Warnf("S3 bucket name mismatch, expected %v, got %v", recordingBucketName, s3.Bucket.Name)
			return
		}

		err := mediaInfo.ExtractMediaInfo(s3.Object.Key)
		if err != nil {
			logger.WithError(err).Errorf("Failed to extract media info for file %v", s3.Object.Key)
		}
	}
}
