package mediainfo

import (
	"bufio"
	"bytes"
	"encoding/json"
	"encoding/xml"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strings"
	"sync"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/sirupsen/logrus"

	mienc "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/mediainfo/encoding"
)

type MediaInfo interface {
	ExtractMediaInfo(s3Key string) error
}

type mediaInfo struct {
	s3Bucket string
	s3Key    string
	s3Client s3iface.S3API
	logger   logrus.FieldLogger
}

func New(s3Bucket string, s3Client s3iface.S3API, logger logrus.FieldLogger) MediaInfo {
	return &mediaInfo{
		s3Bucket: s3Bucket,
		s3Client: s3Client,
		logger:   logger,
	}
}

func (mi *mediaInfo) ExtractMediaInfo(s3Key string) error {
	resp, err := mi.s3Client.GetObject(&s3.GetObjectInput{
		Bucket: aws.String(mi.s3Bucket),
		Key:    aws.String(s3Key),
	})

	if err != nil {
		return err
	}

	tmpFilePath := path.Join("/tmp/", s3Key)

	directory := filepath.Dir(tmpFilePath)

	if _, err := os.Stat(directory); os.IsNotExist(err) {
		os.MkdirAll(directory, 0700)
	}

	var file *os.File

	if file, err = os.Create(tmpFilePath); err != nil {
		return err
	}

	defer file.Close()

	_, err = ioutil.ReadAll(io.TeeReader(resp.Body, file))

	if err != nil {
		return err
	}

	file.Close()

	var wg sync.WaitGroup

	// Basic Media Data
	wg.Add(1)
	go func() {
		defer wg.Done()
		basic, err := mi.execMediaInfo([]string{
			"--output=JSON",
			file.Name(),
		})

		if err != nil {
			mi.logger.WithError(err).Warn("Failed to extract basic data, mediainfo error")
			return
		}

		err = mi.uploadResults(basic, fmt.Sprintf("%v.basic.json", s3Key))

		if err != nil {
			mi.logger.WithError(err).Warn("Failed to upload basic data, S3 error")
			return
		}
	}()

	// MediaTrace data
	wg.Add(1)
	go func() {
		defer wg.Done()
		mediaTraceXML, err := mi.execMediaInfo([]string{
			"--Details=1",
			"--output=XML",   // This mode doesn't support JSON :(
			"--ParseSpeed=1", // Make sure we parse the whole file
			file.Name(),
		})

		if err != nil {
			mi.logger.WithError(err).Warn("Failed to extract trace data, mediainfo error")
			return
		}

		// Convert from XML to JSON
		mediaTrace := mienc.MediaTrace{}

		err = xml.Unmarshal(mediaTraceXML, &mediaTrace)
		if err != nil {
			mi.logger.WithError(err).Warn("Failed to unmarshal trace from XML")
			return
		}

		// We also want to extract just the trace data we want for the GOP structure so we don't
		// Have to download a multi megabyte trace file every time we want to look at the GOP
		wg.Add(1)
		go func() {
			defer wg.Done()
			mi.extractGOPData(mediaTrace, s3Key)
		}()

		mediaTraceJSON, err := json.Marshal(mediaTrace)
		if err != nil {
			mi.logger.WithError(err).Warn("Failed to marshal trace to JSON")
			return
		}

		err = mi.uploadResults(mediaTraceJSON, fmt.Sprintf("%v.trace.json", s3Key))

		if err != nil {
			mi.logger.WithError(err).Warn("Failed to upload trace data, S3 error")
			return
		}
	}()

	wg.Wait()

	err = os.Remove(file.Name())
	if err != nil {
		// This isn't fatal
		mi.logger.WithError(err).Warn("Failed to clean up file")
	}

	return nil
}

func (mi *mediaInfo) extractGOPData(mediaTrace mienc.MediaTrace, s3Key string) {
	if mediaTrace.Media.Blocks == nil {
		mi.logger.Info("No Media found")
		return
	}

	gopData := mienc.GOPData{
		Frames: make([]*mienc.Frame, 0),
	}

	for _, block := range mediaTrace.Media.Blocks {
		if block.Name != "Video" {
			continue
		}

		isVideoFrame := false

		for _, videoData := range block.Data {
			if videoData.Name == "AVCPacketType" && videoData.Info == "NALU" {
				isVideoFrame = true
				break
			}
		}

		if !isVideoFrame {
			continue
		}

		frame := &mienc.Frame{}
		gopData.Frames = append(gopData.Frames, frame)

		for _, videoSubBlock := range block.Blocks {
			if videoSubBlock.Name == "Header" {
				for _, headerData := range videoSubBlock.Data {
					if headerData.Name == "Timestamp_Base" {
						frame.Timestamp = headerData.Value
						break
					}
				}

				continue
			}

			if !strings.HasPrefix(videoSubBlock.Name, "slice_layer_without_partitioning") {
				continue
			}

			frame.Size = videoSubBlock.Size

			for _, sliceBlock := range videoSubBlock.Blocks {
				if sliceBlock.Name != "slice_header" {
					continue
				}

				for _, sliceData := range sliceBlock.Data {
					if sliceData.Name != "slice_type" {
						continue
					}

					frame.PictType = sliceData.Info
					break
				}
			}
		}
	}

	gopJSON, err := json.Marshal(gopData)
	if err != nil {
		mi.logger.WithError(err).Warn("Failed to convert gopData to JSON")
	}

	err = mi.uploadResults(gopJSON, fmt.Sprintf("%v.gop.json", s3Key))

	if err != nil {
		mi.logger.WithError(err).Warn("Failed to upload GOP data, S3 error")
		return
	}
}

func (mi *mediaInfo) uploadResults(data []byte, s3Key string) error {
	_, err := mi.s3Client.PutObject(&s3.PutObjectInput{
		Bucket: aws.String(mi.s3Bucket),
		Key:    aws.String(s3Key),
		Body:   bytes.NewReader(data),
	})

	if err != nil {
		return err
	}

	return nil
}

func (mi *mediaInfo) execMediaInfo(args []string) ([]byte, error) {
	// This relies on mediainfo being available in the PATH
	cmd := exec.Command("mediainfo", args...)

	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return nil, err
	}

	if err := cmd.Start(); err != nil {
		return nil, err
	}

	// read command's stdout line by line
	in := bufio.NewScanner(stdout)

	cmdData := make([]byte, 0)

	for in.Scan() {
		cmdData = append(cmdData, in.Bytes()...)
	}

	if err := cmd.Wait(); err != nil {
		return nil, err
	}

	return cmdData, nil
}
