package mmdb

import (
	"errors"
	"net"

	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	"github.com/oschwald/maxminddb-golang"
	"github.com/sirupsen/logrus"
)

// Client defines the methods that are available in the mmdb client
type Client interface {
	GetIPInfo(ipAddress string) (*csa.IPInfo, error)
}

type client struct {
	ispDBLocation  string
	cityDBLocation string
	logger         logrus.FieldLogger
}

// New creates a new mmdb Client
func New(ispDBLocation, cityDBLocation string, logger logrus.FieldLogger) Client {
	return &client{
		ispDBLocation:  ispDBLocation,
		cityDBLocation: cityDBLocation,
		logger:         logger,
	}
}

func (c *client) GetIPInfo(ipAddress string) (*csa.IPInfo, error) {
	cityDb, err := maxminddb.Open(c.cityDBLocation)
	if err != nil {
		return nil, err
	}

	ispDb, err := maxminddb.Open(c.ispDBLocation)
	if err != nil {
		return nil, err
	}

	ip := net.ParseIP(ipAddress)
	if ip == nil {
		return nil, errors.New("Failed to parse IP address")
	}

	ret := &csa.IPInfo{}

	var locationRecord mmLocationRecord

	err = cityDb.Lookup(ip, &locationRecord)
	if err != nil {
		c.logger.WithError(err).Infof("Could not look up city for IP %v", ip)
	} else {
		ret.Country = locationRecord.Country.Names.EN
		ret.City = locationRecord.City.Names.EN
	}

	var ispRecord mmISPRecord

	err = ispDb.Lookup(ip, &ispRecord)
	if err != nil {
		c.logger.WithError(err).Infof("Could not look up isp for IP %v", ip)
	} else {
		ret.Asn = ispRecord.ASN
		ret.IspName = ispRecord.Organization
	}

	return ret, nil
}
